<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Setup\Patch\Data;

use Amasty\Ccpa\Api\PolicyRepositoryInterface;
use Amasty\Ccpa\Model\Consent\Consent as ConsentModel;
use Amasty\Ccpa\Model\Consent\ConsentStore\ConsentStore;
use Amasty\Ccpa\Model\Consent\Repository;
use Amasty\Ccpa\Model\Policy;
use Amasty\Ccpa\Model\PolicyFactory;
use Amasty\Ccpa\Model\Source\ConsentLinkType;
use Amasty\Ccpa\Model\Source\CountriesRestrictment;
use Magento\Cms\Api\PageRepositoryInterface;
use Magento\Cms\Model\PageFactory;
use Magento\Customer\Model\Customer;
use Magento\Customer\Setup\CustomerSetupFactory;
use Magento\Eav\Model\Entity\Attribute\SetFactory;
use Magento\Eav\Model\Entity\Attribute\Source\Boolean;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ConfigResource\ConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Module\ResourceInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Store\Model\Store;

class InstallSampleData implements DataPatchInterface
{
    public const ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE = 'dont_sell_personal_data';
    public const DEFAULT_CHECKBOX_TEXT = 'I have read and accept the <a href="{link}">privacy policy</a>';

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var PageFactory
     */
    private $pageFactory;

    /**
     * @var PageRepositoryInterface
     */
    private $pageRepository;

    /**
     * @var TypeListInterface
     */
    private $typeList;

    /**
     * @var PolicyRepositoryInterface
     */
    private $policyRepository;

    /**
     * @var PolicyFactory
     */
    private $policyFactory;

    /**
     * @var EavSetupFactory
     */
    private $eavSetupFactory;

    /**
     * @var CustomerSetupFactory
     */
    private $customerSetupFactory;

    /**
     * @var SetFactory
     */
    private $setFactory;

    /**
     * @var Repository
     */
    private $repository;

    /**
     * @var ConfigInterface
     */
    private $resourceConfig;

    /**
     * @var ResourceInterface
     */
    private $moduleResource;

    public function __construct(
        PageFactory $pageFactory,
        PageRepositoryInterface $pageRepository,
        WriterInterface $configWriter,
        TypeListInterface $typeList,
        ConfigInterface $resourceConfig,
        PolicyRepositoryInterface $policyRepository,
        PolicyFactory $policyFactory,
        EavSetupFactory $eavSetupFactory,
        CustomerSetupFactory $customerSetupFactory,
        SetFactory $setFactory,
        Repository $repository,
        ResourceInterface $moduleResource
    ) {
        $this->configWriter = $configWriter;
        $this->pageFactory = $pageFactory;
        $this->pageRepository = $pageRepository;
        $this->typeList = $typeList;
        $this->resourceConfig = $resourceConfig;
        $this->policyRepository = $policyRepository;
        $this->policyFactory = $policyFactory;
        $this->eavSetupFactory = $eavSetupFactory;
        $this->customerSetupFactory = $customerSetupFactory;
        $this->setFactory = $setFactory;
        $this->repository = $repository;
        $this->moduleResource = $moduleResource;
    }

    public function apply()
    {
        $setupDataVersion = $this->moduleResource->getDataVersion('Amasty_Ccpa');

        // Check if module was already installed or not.
        // If setup_version present in DB then we don't need to install fixtures, because setup_version is a marker.
        if (!$setupDataVersion) {
            $path = 'amasty_ccpa/deletion_notification/';
            $fields = ['sender', 'reply_to', 'template'];
            $connection = $this->resourceConfig->getConnection();

            foreach ($fields as $key => $fieldId) {
                $data = ['path' => $path . 'deny_' . $fieldId];
                $whereCondition = ['path = ?' => $path . $fieldId];
                $connection->update($this->resourceConfig->getMainTable(), $data, $whereCondition);
            }

            if (!$this->policyRepository->getCurrentPolicy()) {
                $this->createPolicy();
            }

            $this->createAttribute();
            $this->createConsent();
        }
    }

    public static function getDependencies()
    {
        return [];
    }

    public function getAliases()
    {
        return [];
    }

    /**
     * @throws CouldNotSaveException
     */
    private function createPolicy()
    {
        $policyData = [
            'comment' => 'Automatically generated during installation',
            'policy_version' => 'v1.0',
            'status' => Policy::STATUS_ENABLED,
            'content' => 'This is the privacy policy sample page.' .
                '<br>It was created automatically and do not substitute the one you need to create and ' .
                'provide to your store visitors. <br>Please, replace this text with the correct privacy ' .
                'policy by visiting the Customers > US Privacy Laws > Privacy Policy section in the backend.'
        ];

        /** @var Policy $policyModel */
        $policyModel = $this->policyFactory->create();
        $policyModel->addData($policyData);
        $policyModel->setLastEditedBy(null);

        try {
            $this->policyRepository->save($policyModel);
        } catch (CouldNotSaveException $e) {
            null;
        }
    }

    private function createAttribute()
    {
        $customerSetup = $this->customerSetupFactory->create();

        $customerEntity = $customerSetup->getEavConfig()->getEntityType('customer');
        $attributeSetId = $customerEntity->getDefaultAttributeSetId();

        $attributeSet = $this->setFactory->create();
        $attributeGroupId = $attributeSet->getDefaultGroupId($attributeSetId);
        $customerSetup->addAttribute(
            Customer::ENTITY,
            self::ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE,
            [
                'type'         => 'int',
                'label'        => 'Don\'t Sell or Share My Personal Information',
                'input'        => 'boolean',
                'required'     => false,
                'visible'      => true,
                'user_defined' => true,
                'position'     => 1,
                'system'       => 0,
            ]
        );

        $attribute = $customerSetup->getEavConfig()->getAttribute(
            Customer::ENTITY,
            self::ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE
        )->addData(
            [
                'attribute_set_id' => $attributeSetId,
                'attribute_group_id' => $attributeGroupId,
                'is_filterable_in_grid' => true,
                'is_used_in_grid' => true,
                'source_model' => Boolean::class,
                'used_in_forms' => ['adminhtml_customer']
            ]
        );

        $attribute->save();
    }

    /**
     * @throws CouldNotSaveException
     */
    private function createConsent()
    {
        /** @var ConsentModel $consent * */
        $consent = $this->repository->getEmptyConsentModel();
        $consent->setConsentName(__('Privacy Checkbox')->getText());
        $consent->setConsentCode('privacy_checkbox');

        $consentStore = $this->repository->getEmptyConsentStoreModel();
        $consentStore->addData(
            [
                ConsentStore::CONSENT_STORE_ID => Store::DEFAULT_STORE_ID,
                ConsentStore::IS_ENABLED => false,
                ConsentStore::IS_REQUIRED => true,
                ConsentStore::LOG_THE_CONSENT => true,
                ConsentStore::SORT_ORDER => 0,
                ConsentStore::HIDE_CONSENT_AFTER_USER_LEFT_THE_CONSENT => false,
                ConsentStore::CONSENT_LOCATION => 'registration,checkout,contactus,subscription',
                ConsentStore::CONSENT_TEXT => self::DEFAULT_CHECKBOX_TEXT,
                ConsentStore::LINK_TYPE => ConsentLinkType::PRIVACY_POLICY,
                ConsentStore::VISIBILITY => CountriesRestrictment::EEA_COUNTRIES
            ]
        );
        $consent->setStoreModel($consentStore);

        try {
            $this->repository->save($consent);
        } catch (CouldNotSaveException $e) {
            null;
        }
    }
}
