<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Observer\Customer;

use Amasty\Ccpa\Model\CalifornianDetector;
use Amasty\Ccpa\Model\Config;
use Amasty\Ccpa\Setup\Patch\Data\InstallSampleData;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Customer\Model\Address;
use Magento\Customer\Model\CustomerFactory;
use Magento\Customer\Model\ResourceModel\Customer;
use Magento\Framework\Api\ExtensibleDataObjectConverter;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;

class AfterAddressSave implements ObserverInterface
{
    /**
     * @var CalifornianDetector
     */
    private $californianDetector;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Customer
     */
    private $customerResource;

    /**
     * @var CustomerFactory
     */
    private $customerFactory;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var ExtensibleDataObjectConverter
     */
    private $dataObjectConverter;

    public function __construct(
        CalifornianDetector $californianDetector,
        Config $config,
        CustomerRepositoryInterface $customerRepository,
        Customer $customerResource,
        CustomerFactory $customerFactory,
        ExtensibleDataObjectConverter $dataObjectConverter
    ) {
        $this->californianDetector = $californianDetector;
        $this->config = $config;
        $this->customerResource = $customerResource;
        $this->customerFactory = $customerFactory;
        $this->customerRepository = $customerRepository;
        $this->dataObjectConverter = $dataObjectConverter;
    }

    public function execute(Observer $observer)
    {
        $address = $observer->getCustomerAddress();

        if (!$this->config->isModuleEnabled()
            || !$this->config->isOnlyCalifornians()
            || !$address
        ) {
            return;
        }
        $customerId = (int)$address->getCustomerId();
        $customer = $this->customerRepository->getById($customerId);
        $this->updateAddress($customer, $address);

        if (!$this->californianDetector->isCalifornian($customer)) {
            $customer->setCustomAttribute(
                InstallSampleData::ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE,
                null
            );
            $customerData = $this->dataObjectConverter->toNestedArray($customer, [], CustomerInterface::class);
            $customerModel = $this->customerFactory->create(['data' => $customerData]);
            $customerModel->setId($customer->getId());
            //avoiding save through repository to not run address save again
            $this->customerResource->saveAttribute($customerModel, InstallSampleData::ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE);
        }
    }

    protected function updateAddress(CustomerInterface $customer, Address $newAddress): void
    {
        $addresses = $customer->getAddresses();

        foreach ($addresses as $i => $address) {
            if ($address->getId() == $newAddress->getId()) {
                $addresses[$i] = $newAddress->getDataModel();
            }
        }
        $customer->setAddresses($addresses);
    }
}
