<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Model;

use Amasty\Ccpa\Api\Data\WithConsentInterface;
use Amasty\Ccpa\Model\ResourceModel\WithConsent\CollectionFactory;
use Amasty\Geoip\Model\Geolocation;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\HTTP\PhpEnvironment\RemoteAddress;
use Magento\Store\Model\StoreManagerInterface;

class Visitor
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var Geolocation
     */
    private $geolocation;

    /**
     * @var RemoteAddress
     */
    private $remoteAddress;

    /**
     * @var CollectionFactory
     */
    private $withConsentCollectionFactory;

    /**
     * @var PolicyRepository
     */
    private $policyRepository;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    public function __construct(
        Config $config,
        CustomerSession $customerSession,
        CheckoutSession $checkoutSession,
        Geolocation $geolocation,
        RemoteAddress $remoteAddress,
        CollectionFactory $withConsentCollectionFactory,
        PolicyRepository $policyRepository,
        StoreManagerInterface $storeManager
    ) {
        $this->config = $config;
        $this->customerSession = $customerSession;
        $this->checkoutSession = $checkoutSession;
        $this->geolocation = $geolocation;
        $this->remoteAddress = $remoteAddress;
        $this->withConsentCollectionFactory = $withConsentCollectionFactory;
        $this->policyRepository = $policyRepository;
        $this->storeManager = $storeManager;
    }

    /**
     * @return bool|string
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCountryCode()
    {
        $customer = $this->customerSession->getCustomer();

        if ($countryCode = $this->checkoutSession->getQuote()->getShippingAddress()->getCountry()) {
            return (string)$countryCode;
        }

        if ($countryCode = $this->checkoutSession->getQuote()->getBillingAddress()->getCountry()) {
            return (string)$countryCode;
        }

        if ($customer && ($address = $customer->getPrimaryBillingAddress())) {
            if ($countryCode = $address->getCountry()) {
                return (string)$countryCode;
            }
        }

        if ($countryCode = $this->locate()) {
            return (string)$countryCode;
        }

        return false;
    }

    protected function locate()
    {
        if ($this->customerSession->hasData('amccpa_country')) {
            return $this->customerSession->getData('amccpa_country');
        }

        $geolocationResult = $this->geolocation->locate($this->getRemoteIp());

        $result = isset($geolocationResult['country']) ? $geolocationResult['country'] : false;

        $this->customerSession->setData('amccpa_country', $result);

        return $result;
    }

    public function getRemoteIp()
    {
        $ip = $this->remoteAddress->getRemoteAddress();
        $ip = substr($ip, 0, strrpos($ip, ".")) . '.0';

        return $ip;
    }

    protected function isEEACountry($countryCode)
    {
        return in_array($countryCode, $this->config->getEEACountryCodes());
    }

    /**
     * Getting already agreed consents
     *
     * @return array
     */
    public function getAgreedConsents()
    {
        if (!$this->customerSession->isSessionExists()) {
            $this->customerSession->start();
        }
        $customerId = $this->customerSession->getCustomerId();
        $storeId = $this->storeManager->getStore()->getId();
        $currentPolicy = $this->policyRepository->getCurrentPolicy($storeId);

        if ($customerId && $storeId) {
            $consentLogCollection = $this->withConsentCollectionFactory->create();
            return $consentLogCollection->addFieldToFilter(WithConsentInterface::CUSTOMER_ID, $customerId)
                ->addFieldToFilter(WithConsentInterface::ACTION, true)
                ->addFieldToFilter(WithConsentInterface::POLICY_VERSION, $currentPolicy->getPolicyVersion())
                ->getColumnValues(WithConsentInterface::CONSENT_CODE);
        }

        return [];
    }
}
