<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Model;

use Amasty\Ccpa\Api\PolicyRepositoryInterface;
use Amasty\Ccpa\Api\WithConsentRepositoryInterface;
use Amasty\Ccpa\Model\ResourceModel\WithConsent as WithConsentResource;
use Amasty\Ccpa\Model\Source\ConsentLinkType;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class ConsentLogger
{
    public const FROM_REGISTRATION = 'registration';

    public const FROM_CHECKOUT = 'checkout';

    public const FROM_SUBSCRIPTION = 'subscription';

    public const FROM_CONTACTUS = 'contactus';

    public const FROM_EMAIL = 'email';

    public const CMS_PAGE = 'CMS Page';

    /**
     * @var WithConsentRepositoryInterface
     */
    private $withConsentRepository;

    /**
     * @var WithConsentFactory
     */
    private $consentFactory;

    /**
     * @var PolicyRepositoryInterface
     */
    private $policyRepository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var WithConsentResource
     */
    private $withConsent;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Amasty\Ccpa\Model\Visitor
     */
    private $visitor;

    public function __construct(
        WithConsentRepositoryInterface $withConsentRepository,
        WithConsentFactory $consentFactory,
        PolicyRepositoryInterface $policyRepository,
        ActionLogger $logger,
        WithConsentResource $withConsent,
        StoreManagerInterface $storeManager,
        Visitor $visitor
    ) {
        $this->withConsentRepository = $withConsentRepository;
        $this->consentFactory = $consentFactory;
        $this->policyRepository = $policyRepository;
        $this->logger = $logger;
        $this->withConsent = $withConsent;
        $this->storeManager = $storeManager;
        $this->visitor = $visitor;
    }

    /**
     * @param string|int $customerId
     * @param string $from
     * @param Consent\Consent $consentModel
     *
     * @return void
     * @throws LocalizedException
     */
    public function log($customerId, $from, $consentModel = null)
    {
        if (!$customerId) {
            return;
        }

        if ($policy = $this->policyRepository->getCurrentPolicy()) {
            try {
                /** @var WithConsent $withConsent */
                $withConsent = $this->consentFactory->create();
                $privacyPolicyVersionValue = $consentModel->getPrivacyLinkType() === ConsentLinkType::PRIVACY_POLICY ?
                    $policy->getPolicyVersion() : self::CMS_PAGE;
                $withConsent->setPolicyVersion($privacyPolicyVersionValue);
                $withConsent->setGotFrom($from);
                $withConsent->setWebsiteId($this->storeManager->getWebsite()->getId());
                $withConsent->setIp($this->visitor->getRemoteIp());
                $withConsent->setCustomerId($customerId);
                $withConsent->setAction($consentModel->isConsentAccepted());
                $withConsent->setConsentCode($consentModel->getConsentCode());

                $this->withConsentRepository->save($withConsent);
            } catch (\Exception $exception) {
                $this->logger->critical($exception);
            }
        }
    }
}
