<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Model\Consent\DataProvider;

use Amasty\Ccpa\Model\Config;
use Amasty\Ccpa\Model\Consent\Consent;
use Amasty\Ccpa\Model\Consent\ConsentStore\ConsentStore;
use Amasty\Ccpa\Model\Consent\ResourceModel\Collection;
use Amasty\Ccpa\Model\Consent\ResourceModel\CollectionFactory;
use Amasty\Ccpa\Model\Source\CountriesRestrictment;
use Amasty\Ccpa\Model\Visitor;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

abstract class AbstractDataProvider
{
    /**
     * @var Config
     */
    protected $config;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Visitor
     */
    protected $visitor;

    public function __construct(
        Config $config,
        CollectionFactory $collectionFactory,
        StoreManagerInterface $storeManager,
        Visitor $visitor
    ) {
        $this->config = $config;
        $this->collectionFactory = $collectionFactory;
        $this->storeManager = $storeManager;
        $this->visitor = $visitor;
    }

    /**
     * @param string $location
     *
     * @return array
     */
    abstract public function getData(string $location);

    /**
     * @param string $location
     *
     * @return Collection
     * @throws NoSuchEntityException
     */
    public function getConsentCollection($location)
    {
        /** @var Collection $collection **/
        $collection = $this->collectionFactory->create();
        $storeId = (int)$this->storeManager->getStore()->getId();
        $collection
            ->addStoreData($storeId)
            ->addMultiValueFilter(ConsentStore::CONSENT_LOCATION, $location)
            ->addFieldToFilter(ConsentStore::IS_ENABLED, true)
            ->addOrder(ConsentStore::SORT_ORDER, Collection::SORT_ORDER_ASC);

        return array_filter($collection->getItems(), function ($consent) {
            return $this->isNeedShowConsent($consent);
        });
    }

    /**
     * @param Consent $consent
     *
     * @return bool
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function isNeedShowConsent(Consent $consent)
    {
        return $this->isNeedShowConsentByCountry($consent) && $this->isNeedShowConsentByAgreement($consent);
    }

    /**
     * @param Consent $consent
     *
     * @return bool
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function isNeedShowConsentByCountry(Consent $consent)
    {
        $countries = $consent->getCountries() ?: [];
        $country = $this->visitor->getCountryCode();

        switch ($consent->getVisibility()) {
            case CountriesRestrictment::ALL_COUNTRIES:
                return true;
            case CountriesRestrictment::EEA_COUNTRIES:
                return in_array(
                    $country,
                    $this->config->getEuCountriesCodes()
                );
            case CountriesRestrictment::SPECIFIED_COUNTRIES:
                return in_array(
                    $country,
                    $countries
                );
        }

        return false;
    }

    /**
     * @param Consent $consent
     *
     * @return bool
     * @throws LocalizedException
     */
    public function isNeedShowConsentByAgreement(Consent $consent)
    {
        if ($consent->isHideTheConsentAfterUserLeftTheConsent()) {
            $agreedConsents = $this->visitor->getAgreedConsents();

            return !in_array($consent->getConsentCode(), $agreedConsents);
        } else {
            return true;
        }
    }
}
