<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Store\Model\ScopeInterface;

class Config
{
    public const PATH_PREFIX = 'amasty_ccpa';

    /**#@+
     * Constants defined for xpath of system configuration
     */
    public const PRIVACY_CHECKBOX_EEA_COUNTRIES = 'privacy_checkbox/eea_countries';

    public const ONLY_CALIFORNIANS = 'general/only_californians';

    public const AUTO_CLEANING = 'general/auto_cleaning';

    public const AUTO_CLEANING_DAYS = 'general/auto_cleaning_days';

    public const AVOID_ANONYMIZATION = 'general/avoid_anonymisation';

    public const ORDER_STATUSES = 'general/order_statuses';

    public const AVOID_GIFT_REGISTRY_ANONYMIZATION = 'general/gift_registry_anonymisation';

    public const NOTIFICATE_ADMIN = 'deletion_notification/enable_admin_notification';

    public const NOTIFICATE_ADMIN_TEMPLATE = 'deletion_notification/admin_template';

    public const NOTIFICATE_ADMIN_SENDER = 'deletion_notification/admin_sender';

    public const NOTIFICATE_ADMIN_RECIEVER = 'deletion_notification/admin_reciever';

    public const EMAIL_NOTIFICATION_TEMPLATE = '_notification/template';

    public const EMAIL_NOTIFICATION_SENDER = '_notification/sender';

    public const EMAIL_NOTIFICATION_REPLY_TO = '_notification/reply_to';

    public const ALLOWED = 'customer_access_control/';

    public const DOWNLOAD = 'download';

    public const ALLOW_OPT_OUT = 'allow_opt_out';

    public const DELETE = 'delete';

    public const MODULE_ENABLED = 'general/enabled';

    public const EU_COUNTRIES = 'general/country/eu_countries';

    public const PERSONAL_DATA_DELETION = 'personal_data/automatic_personal_data_deletion/personal_data_deletion';

    public const PERSONAL_DATA_DELETION_DAYS =
        'personal_data/automatic_personal_data_deletion/personal_data_deletion_days';

    public const PERSONAL_DATA_STORED = 'personal_data/anonymization_data/personal_data_stored';

    public const PERSONAL_DATA_STORED_DAYS = 'personal_data/anonymization_data/personal_data_stored_days';

    /**#@-*/

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var DateTime
     */
    private $dateTime;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        DateTime $dateTime
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->dateTime = $dateTime;
    }

    /**
     * @return array
     */
    public function getEEACountryCodes()
    {
        $codes = explode(',', $this->getValue(self::PRIVACY_CHECKBOX_EEA_COUNTRIES));

        return $codes;
    }

    /**
     * An alias for scope config with default scope type SCOPE_STORE
     *
     * @param string $key
     * @param string|null $scopeCode
     * @param string $scopeType
     *
     * @return string|null
     */
    public function getValue($key, $scopeCode = null, $scopeType = ScopeInterface::SCOPE_STORE)
    {
        return $this->scopeConfig->getValue(self::PATH_PREFIX . '/' . $key, $scopeType, $scopeCode);
    }

    /**
     * @param string $path
     * @param string|null $scopeCode
     * @param string $scopeType
     *
     * @return bool
     */
    public function isSetFlag($path, $scopeCode = null, $scopeType = ScopeInterface::SCOPE_STORE)
    {
        return $this->scopeConfig->isSetFlag(self::PATH_PREFIX . '/' . $path, $scopeType, $scopeCode);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function isAdminDeleteNotificationEnabled($scopeCode = null)
    {
        return (bool)$this->getValue(self::NOTIFICATE_ADMIN, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getAdminNotificationTemplate($scopeCode = null)
    {
        return $this->getValue(self::NOTIFICATE_ADMIN_TEMPLATE, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getAdminNotificationSender($scopeCode = null)
    {
        return $this->getValue(self::NOTIFICATE_ADMIN_SENDER, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getAdminNotificationReciever($scopeCode = null)
    {
        return $this->getValue(self::NOTIFICATE_ADMIN_RECIEVER, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null|string $scopeCode
     * @return bool
     */
    public function isOnlyCalifornians($scopeCode = null)
    {
        return $this->isSetFlag(self::ONLY_CALIFORNIANS, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @return bool
     */
    public function isAutoCleaning()
    {
        return $this->isSetFlag(self::AUTO_CLEANING);
    }

    /**
     * @return int
     */
    public function getAutoCleaningDays()
    {
        return (int)$this->getValue(self::AUTO_CLEANING_DAYS);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return bool
     */
    public function isAvoidAnonymization($scopeCode = null)
    {
        return $this->isSetFlag(self::AVOID_ANONYMIZATION, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getOrderStatuses($scopeCode = null)
    {
        return $this->getValue(self::ORDER_STATUSES, $scopeCode, ScopeInterface::SCOPE_WEBSITE);
    }

    /**
     * @param null $scopeCode
     *
     * @return string|null
     */
    public function isAvoidGiftRegistryAnonymization($scopeCode = null)
    {
        return $this->getValue(
            self::AVOID_GIFT_REGISTRY_ANONYMIZATION,
            $scopeCode,
            ScopeInterface::SCOPE_WEBSITE
        );
    }

    /**
     * @param string      $configPath
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getConfirmationEmailTemplate($configPath, $scopeCode = null)
    {
        return $this->getValue(
            $configPath . self::EMAIL_NOTIFICATION_TEMPLATE,
            $scopeCode,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @param string      $configPath
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getConfirmationEmailSender($configPath, $scopeCode = null)
    {
        return $this->getValue($configPath . self::EMAIL_NOTIFICATION_SENDER, $scopeCode, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @param string      $configPath
     * @param null|string $scopeCode
     *
     * @return null|string
     */
    public function getConfirmationEmailReplyTo($configPath, $scopeCode = null)
    {
        $value = $this->getValue(
            $configPath . self::EMAIL_NOTIFICATION_REPLY_TO,
            $scopeCode,
            ScopeInterface::SCOPE_STORE
        ) ?? '';

        return trim($value);
    }

    /**
     * @param string      $configPath
     * @param null|string $scopeCode
     *
     * @return bool
     */
    public function isAllowed($configPath, $scopeCode = null)
    {
        return $this->isSetFlag(self::ALLOWED . $configPath, $scopeCode, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @param null|string $scopeCode
     *
     * @return bool
     */
    public function isModuleEnabled($scopeCode = null)
    {
        return $this->isSetFlag(self::MODULE_ENABLED, $scopeCode, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return array
     */
    public function getEuCountriesCodes()
    {
        $countriesCodes = $this->scopeConfig->getValue(self::EU_COUNTRIES) ?: '';

        return explode(',', $countriesCodes);
    }

    /**
     * @return bool
     */
    public function isPersonalDataDeletion()
    {
        return $this->isSetFlag(self::PERSONAL_DATA_DELETION);
    }

    /**
     * @return int
     */
    public function getPersonalDataDeletionDays()
    {
        return (int)$this->getValue(self::PERSONAL_DATA_DELETION_DAYS);
    }

    /**
     * @return bool
     */
    public function isPersonalDataStored()
    {
        return $this->isSetFlag(self::PERSONAL_DATA_STORED);
    }

    /**
     * @return int
     */
    public function getPersonalDataStoredDays()
    {
        return (int)$this->getValue(self::PERSONAL_DATA_STORED_DAYS);
    }

    /**
     * @return bool
     */
    public function isAnySectionVisible()
    {
        return $this->isAllowed(self::DOWNLOAD)
            || $this->isAllowed(self::ALLOW_OPT_OUT)
            || $this->isAllowed(self::DELETE);
    }
}
