<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Controller\Customer;

use Amasty\Ccpa\Api\DeleteRequestRepositoryInterface;
use Amasty\Ccpa\Model\ActionLogger;
use Amasty\Ccpa\Model\Config\Source\Action as ActionOptions;
use Amasty\Ccpa\Model\DeleteRequest\DeleteRequestSource;
use Amasty\Ccpa\Model\DeleteRequest;
use Amasty\Ccpa\Model\DeleteRequestFactory;
use Magento\Customer\Controller\AbstractAccount as AbstractAccountAction;
use Magento\Framework\App\Action\Context;
use Magento\Customer\Model\Session;
use Psr\Log\LoggerInterface;
use Magento\Framework\Data\Form\FormKey\Validator as FormKeyValidator;
use Magento\Customer\Model\Authentication;
use Amasty\Ccpa\Model\Config;
use Amasty\Ccpa\Model\ResourceModel\DeleteRequest\CollectionFactory;
use Amasty\Ccpa\Model\DeleteRequest\Notifier;

class AddDeleteRequest extends AbstractAccountAction
{
    /**
     * @var \Magento\Customer\Model\Session
     */
    private $customerSession;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var DeleteRequestRepositoryInterface
     */
    private $deleteRequestRepository;

    /**
     * @var DeleteRequestFactory
     */
    private $deleteRequestFactory;

    /**
     * @var ActionLogger
     */
    private $actionLogger;

    /**
     * @var FormKeyValidator
     */
    private $formKeyValidator;

    /**
     * @var Authentication
     */
    private $authentication;

    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var CollectionFactory
     */
    private $deleteRequestCollectionFactory;

    /**
     * @var Notifier
     */
    private $notifier;

    public function __construct(
        Context $context,
        Session $customerSession,
        LoggerInterface $logger,
        DeleteRequestFactory $deleteRequestFactory,
        DeleteRequestRepositoryInterface $deleteRequestRepository,
        ActionLogger $actionLogger,
        FormKeyValidator $formKeyValidator,
        Authentication $authentication,
        Config $configProvider,
        CollectionFactory $deleteRequestCollectionFactory,
        Notifier $notifier
    ) {
        parent::__construct($context);
        $this->customerSession = $customerSession;
        $this->logger = $logger;
        $this->deleteRequestRepository = $deleteRequestRepository;
        $this->deleteRequestFactory = $deleteRequestFactory;
        $this->actionLogger = $actionLogger;
        $this->formKeyValidator = $formKeyValidator;
        $this->authentication = $authentication;
        $this->configProvider = $configProvider;
        $this->deleteRequestCollectionFactory = $deleteRequestCollectionFactory;
        $this->notifier = $notifier;
    }

    public function execute()
    {
        $errorMessage = '';

        if (!$this->formKeyValidator->validate($this->getRequest())) {
            $errorMessage = __('Invalid Form Key. Please refresh the page.');
        }

        if (!$this->configProvider->isAllowed(Config::DELETE)) {
            $errorMessage = __('Access denied.');
        }

        if ($errorMessage) {
            $this->messageManager->addErrorMessage($errorMessage);
            return $this->resultRedirectFactory->create()->setPath('*/*/settings');
        }

        $customerId = $this->customerSession->getCustomerId();
        $customerPass = $this->getRequest()->getParam('current_password');

        try {
            $this->authentication->authenticate($customerId, $customerPass);
        } catch (\Magento\Framework\Exception\AuthenticationException $e) {
            $this->messageManager->addErrorMessage(__('Wrong Password. Please recheck it.'));
            return $this->resultRedirectFactory->create()->setPath('*/*/settings');
        }

        try {
            $deleteRequests = $this->deleteRequestCollectionFactory->create();

            if ($deleteRequests->addFieldToFilter('customer_id', $customerId)->getSize()) {
                $this->messageManager->addErrorMessage(
                    __('Your delete account request is awaiting for the review by the administrator.')
                );
            } else {
                /** @var DeleteRequest $request */
                $request = $this->deleteRequestFactory->create();
                $request->setCustomerId($customerId);
                $request->setGotFrom(DeleteRequestSource::CUSTOMER_REQUEST);
                $this->deleteRequestRepository->save($request);
                $this->actionLogger->logAction(ActionOptions::DELETE_REQUEST_SUBMITTED, $request->getCustomerId());
                if ($this->configProvider->isAdminDeleteNotificationEnabled()) {
                    $this->notifier->notifyAdmin($customerId);
                }
                $this->messageManager->addSuccessMessage(
                    __('Thank you, your account delete request was recorded.')
                );
            }
        } catch (\Exception $exception) {
            $this->messageManager->addErrorMessage(__('An error has occurred'));
            $this->logger->critical($exception);
        }

        return $this->resultRedirectFactory->create()->setPath('*/*/settings');
    }
}
