<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package California Consumer Privacy Act for Magento 2
 */

namespace Amasty\Ccpa\Block;

use Amasty\Ccpa\Model\Config;
use Amasty\Ccpa\Setup\Patch\Data\InstallSampleData;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\Session;
use Magento\Framework\Data\Form\FormKey as FormKey;
use Magento\Framework\View\Element\Template;

class Settings extends Template
{
    public const DOWNLOAD_DATA_BLOCK_SHORT_NAME = 'download';
    public const ALLOW_OPT_OUT_ACCOUNT_PAGE = 'allow_opt_oup_account_page';
    public const DELETE_ACCOUT_BLOCK_SHORT_NAME = 'delete';
    public const VISIBLE_BLOCK_LAYOUT_VARIABLE_NAME = 'visible_block';
    public const NEED_PASSWORD_LAYOUT_VARIABLE_NAME = 'need_password';
    public const IS_ORDER_LAYOUT_VARIABLE_NAME = 'is_order';

    /**
     * @var string
     */
    protected $_template = 'settings.phtml';

    /**
     * @var FormKey
     */
    protected $formKey;

    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var Session
     */
    private $session;

    public function __construct(
        Template\Context $context,
        \Magento\Framework\Registry $registry,
        FormKey $formKey,
        Config $configProvider,
        CustomerRepositoryInterface $customerRepository,
        Session $session,
        array $data = []
    ) {
        parent::__construct($context);
        $this->formKey = $formKey;
        $this->registry = $registry;
        $this->configProvider = $configProvider;
        $this->customerRepository = $customerRepository;
        $this->session = $session;
    }

    /**
     * @return array
     */
    public function getPrivacySettings()
    {
        return $this->getAvailableBlocks();
    }

    /**
     * @return array
     */
    private function getPrivacyBlocks()
    {
        $result = [];

        if ($this->configProvider->isModuleEnabled()) {
            if ($this->configProvider->isAllowed(Config::DOWNLOAD)) {
                $result[self::DOWNLOAD_DATA_BLOCK_SHORT_NAME] = [
                    'title' => __('Download personal data'),
                    'content' => __('Here you can download a copy of your personal data which' .
                        ' we store for your account in CSV format.'),
                    'type' => false,
                    'checkboxText' => '',
                    'hidePassword' => false,
                    'needPassword' => $this->isNeedPassword(),
                    'submitText' => __('Download'),
                    'action' => $this->getUrl('ccpa/customer/downloadCsv'),
                    'actionCode' => Config::DOWNLOAD
                ];
            }

            if ($this->configProvider->isAllowed(Config::DELETE)) {
                $result[self::DELETE_ACCOUT_BLOCK_SHORT_NAME] = [
                    'title' => __('Delete account'),
                    'content' => __('Request to remove your account, together with all your personal data, will be ' .
                        'processed by our staff.<br>Deleting your account will remove all the purchase history, ' .
                        'discounts, orders, invoices and all other information that might be related to your account ' .
                        'or your purchases.<br>All your orders and similar information will be lost.<br>You will not ' .
                        'be able to restore access to your account after we approve your removal request.'),
                    'checked' => true,
                    'type' => 'checkbox',
                    'checkboxText' => __('I understand and I want to delete my account'),
                    'hidePassword' => true,
                    'needPassword' => $this->isNeedPassword(),
                    'submitText' => __('Submit request'),
                    'action' => $this->getUrl('ccpa/customer/addDeleteRequest'),
                    'actionCode' => Config::DELETE
                ];
            }

            if ($this->configProvider->isAllowed(Config::ALLOW_OPT_OUT)) {
                $customerId = $this->session->getCustomerId();
                $allowOptOutAttribute = $this->customerRepository->getById($customerId)->getCustomAttribute(
                    InstallSampleData::ALLOW_OPT_OUT_CUSTOMER_ATTRIBUTE
                );
                $result[self::ALLOW_OPT_OUT_ACCOUNT_PAGE] = [
                    'title' => __('Don\'t Sell or Share My Personal Information'),
                    'content' => __('The US Privacy Laws provides residents of the states with the right to'
                        . ' opt-out and stop businesses from selling or sharing their personal information. Do you'
                        . ' want to stop businesses from selling or sharing your personal information?'),
                    'checked' => true,
                    'value' => $allowOptOutAttribute ? $allowOptOutAttribute->getValue() : 0,
                    'type' => 'select',
                    'hidePassword' => false,
                    'needPassword' => $this->isNeedPassword(),
                    'submitText' => __('Save attribute'),
                    'action' => $this->getUrl('ccpa/customer/saveCustomer'),
                    'actionCode' => Config::ALLOW_OPT_OUT
                ];
            }
        }

        return $result;
    }

    /**
     * @return array
     */
    private function getAvailableBlocks()
    {
        $result = [];
        $allBlocks = $this->getPrivacyBlocks();
        $visibleBlocks = $this->getData(self::VISIBLE_BLOCK_LAYOUT_VARIABLE_NAME)
            ? explode(',', $this->getData(self::VISIBLE_BLOCK_LAYOUT_VARIABLE_NAME)) : [];

        if (!$visibleBlocks) {
            return $allBlocks;
        }

        foreach ($visibleBlocks as $blockName) {
            if (array_key_exists($blockName, $allBlocks)) {
                $result[$blockName] = $allBlocks[$blockName];
            }
        }

        return $result;
    }

    /**
     * @return bool
     */
    public function isNeedPassword()
    {
        return $this->getData(self::NEED_PASSWORD_LAYOUT_VARIABLE_NAME);
    }

    /**
     * @return int
     */
    public function isOrder()
    {
        return (int) $this->getData(self::IS_ORDER_LAYOUT_VARIABLE_NAME);
    }

    /**
     * @return string|int
     */
    public function getCurrentOderIncrementId()
    {
        $currentOrder = $this->registry->registry('current_order');

        return $currentOrder ? $currentOrder->getIncrementId() : null;
    }

    /**
     * Retrieve Session Form Key
     *
     * @return string
     */
    public function getFormKey()
    {
        return $this->formKey->getFormKey();
    }
}
