<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Blog Pro Hyva Compatibility by Amasty
 */

declare(strict_types=1);

namespace Amasty\BlogHyva\ViewModel;

use Magento\Framework\View\Element\Block\ArgumentInterface;
use Amasty\Blog\Model\Blog\Registry;
use Amasty\Blog\Helper\Settings;
use Amasty\Blog\Api\CommentRepositoryInterface;
use Amasty\Blog\Block\Comments\Message;
use Amasty\Blog\Model\Comments as CommentModel;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Template;
use Magento\Framework\Session\SessionManager;

class Post implements ArgumentInterface
{
    /**
     * @var Registry $registry
     */
    private $registry;

    /**
     * @var Settings $settingsHelper
     */
    private $settingsHelper;

    /**
     * @var Template $template
     */
    private $template;

    /**
     * @var CommentRepositoryInterface $commentRepository
     */
    private $commentRepository;

    /**
     * @var UrlInterface $urlBuilder
     */
    private $urlBuilder;

    /**
     * @var SessionManager $session
     */
    protected $session;

    /**
     * @param Registry $registry
     * @param Settings $settingsHelper
     * @param CommentRepositoryInterface $commentRepository
     * @param UrlInterface $urlBuilder
     * @param Template $template
     * @param SessionManager $session
     */
    public function __construct(
        Registry $registry,
        Settings $settingsHelper,
        CommentRepositoryInterface $commentRepository,
        UrlInterface $urlBuilder,
        Template $template,
        SessionManager $session
    ) {
        $this->registry = $registry;
        $this->settingsHelper = $settingsHelper;
        $this->commentRepository = $commentRepository;
        $this->urlBuilder = $urlBuilder;
        $this->template = $template;
        $this->session = $session;
    }

    /**
     * Return posts
     *
     * @return \Amasty\Blog\Model\Posts|bool|mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getPost(): mixed
    {
        $result = false;
        $parent = $this->template->getParentBlock();

        if ($parent) {
            if ($parent instanceof \Amasty\Blog\Block\Content\Post) {
                $result = $parent->getPost();
            }
        }

        if (!$result) {
            $result = $this->template->getData('post') ?: $this->registry->registry(Registry::CURRENT_POST);
        }

        return $result;
    }

    /**
     * Return post ID
     *
     * @return int
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getPostId(): int
    {
        return (int)$this->getPost()->getId();
    }

    /**
     * Get Comment Message HTML
     *
     * @param CommentModel $message
     *
     * @return bool|string
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getMessageHtml(CommentModel $message)
    {
        $result = false;
        $messageBlock = $this->template->getLayout()->getBlock(Message::AMBLOG_COMMENTS_MESSAGE);
        if (!$messageBlock) {
            $messageBlock = $this->template->getLayout()
                ->createBlock(\Amasty\Blog\Block\Comments\Message::class, Message::AMBLOG_COMMENTS_MESSAGE)
                ->setTemplate("Amasty_Blog::comments/list/message.phtml");
        }
        if ($messageBlock) {
            $messageBlock->setMessage($message);
            $result = $messageBlock->toHtml();
        }

        return $result;
    }

    /**
     * Get Comments Collection
     *
     * @return \Amasty\Blog\Model\ResourceModel\Comments\Collection
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getCollection(): mixed
    {
        $comments = $this->commentRepository->getCommentsInPost($this->getPostId());

        $comments->addActiveFilter(
            $this->settingsHelper->getCommentsAutoapprove()
                ? null
                : $this->getSessionId()
        );

        $comments->setDateOrder(\Magento\Framework\DB\Select::SQL_ASC)->setNotReplies();

        return $comments;
    }

    /**
     * Return comment form url
     *
     * @return string
     */
    public function getFormUrl(): string
    {
        return $this->urlBuilder->getUrl(
            'amblog/index/form',
            [
                'reply_to' => '{{reply_to}}',
                'post_id' => '{{post_id}}',
                'session_id' => '{{session_id}}',
            ]
        );
    }

    /**
     * Return comment update form url
     *
     * @return string
     */
    public function getUpdateUrl(): string
    {
        return $this->urlBuilder->getUrl('amblog/index/updateComments');
    }

    /**
     * Return post url
     *
     * @return string
     */
    public function getPostUrl(): string
    {
        return $this->urlBuilder->getUrl(
            'amblog/index/postForm',
            [
                'reply_to' => '{{reply_to}}',
                'post_id' => '{{post_id}}',
            ]
        );
    }

    /**
     * Return session ID
     *
     * @return string
     */
    public function getSessionId(): string
    {
        return $this->session->getSessionId();
    }

    /**
     * Return fetchView
     *
     * @param Sting $filePath
     * @return mixed
     */
    public function fetchView($filePath): mixed
    {
        return $this->template->fetchView($filePath);
    }

    /**
     * Return fetchView
     *
     * @param Sting $filePath
     * @return string
     */
    public function getTemplateFile($filePath): string
    {
        return $this->template->getTemplateFile($filePath);
    }
}
