<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Auto Add Promo Items Graphql by Amasty
 */

namespace Amasty\PromoGraphql\Model\Resolver;

use Amasty\Promo\Helper\Cart;
use Amasty\Promo\Model\ItemRegistry\PromoItemData;
use Amasty\Promo\Model\PromoItemRepository;
use Amasty\PromoGraphql\Model\Di\Wrapper as DiWrapper;
use Amasty\PromoGraphql\Model\QuoteProcessor;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Message\MessageInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Cart\BuyRequest\BuyRequestBuilder;
use Magento\Quote\Model\Cart\Data\CartItemFactory;

class AddFreeGifts implements ResolverInterface
{
    /**
     * @var CartRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var PromoItemRepository
     */
    private $promoItemRepository;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var Cart
     */
    private $promoCartHelper;

    /**
     * @var QuoteProcessor
     */
    private $quoteProcessor;

    /**
     * use DiWrapper for compatibility with m2.3, where class BuyRequestBuilder doesn't exist.
     *
     * @var DiWrapper|BuyRequestBuilder
     */
    private $requestBuilder;

    /**
     * use DiWrapper for compatibility with m2.3, where class CartItemFactory doesn't exist.
     *
     * @var DiWrapper|CartItemFactory
     */
    private $cartItemFactory;

    public function __construct(
        QuoteProcessor $quoteProcessor,
        CartRepositoryInterface $quoteRepository,
        PromoItemRepository $promoItemRepository,
        ProductRepositoryInterface $productRepository,
        Cart $promoCartHelper,
        DiWrapper $cartItemFactory,
        DiWrapper $requestBuilder
    ) {
        $this->quoteProcessor = $quoteProcessor;
        $this->quoteRepository = $quoteRepository;
        $this->promoItemRepository = $promoItemRepository;
        $this->productRepository = $productRepository;
        $this->promoCartHelper = $promoCartHelper;
        $this->cartItemFactory = $cartItemFactory;
        $this->requestBuilder = $requestBuilder;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $quote = $this->quoteProcessor->process($args['cartId'], $context);
        $updateTotalQty = false;
        $allErrors = [];

        foreach ($args['promoItems'] as $promoItem) {
            $promoItemData = $this->getPromoDataItem($promoItem['sku'], (int)$quote->getId());

            if (!$promoItemData) {
                $allErrors[] = __('Free gift %1 wasn\'t added.', $promoItem['sku']);
                continue;
            }

            $product = $this->productRepository->get($promoItem['sku']);
            try {
                $qty = min($promoItem['quantity'], $promoItemData->getQtyToProcess());
                $requestParams = $this->resolveRequestParams($promoItem);

                $addItemResult = $this->promoCartHelper->addProduct(
                    $product,
                    $qty,
                    $promoItemData,
                    $requestParams,
                    $quote
                );
                if (!$addItemResult) {
                    $allErrors[] = __('Free gift %1 wasn\'t added.', $promoItem['sku']);
                } else {
                    $updateTotalQty = true;
                }
            } catch (\Exception $e) {
                $allErrors[] = $e->getMessage();
            }
        }

        if ($updateTotalQty) {
            $this->quoteRepository->save($quote);
        }

        if ($quote->getData('has_error')) {
            $errors = $quote->getErrors();

            /** @var MessageInterface $error */
            foreach ($errors as $error) {
                $allErrors[] = $error->getText();
            }
        }

        return [
            'cart' => [
                'model' => $quote,
            ],
            'user_errors' => $allErrors
        ];
    }

    private function getPromoDataItem(string $sku, int $quoteId): ?PromoItemData
    {
        $promoItemsGroup = $this->promoItemRepository->getItemsByQuoteId($quoteId);
        $promoItemsData = $promoItemsGroup->getItemsBySku($sku);
        foreach ($promoItemsData as $promoItemData) {
            if ($promoItemData->getQtyToProcess() > 0) {
                return $promoItemData;
            }
        }

        return null;
    }

    private function resolveRequestParams(array $promoItem): array
    {
        $cartItem = $this->cartItemFactory->create($promoItem);
        $requestParams = $this->requestBuilder->build($cartItem);

        return (is_object($requestParams)) ? $requestParams->getData() : [];
    }
}
