/**
 * Amasty custom gallery - init slider listeners
 * and options for the built scope
 */

define([
    'jquery',
    'underscore',
    'Amasty_Gallery/js/model/am-gallery-model'
], function ($, _, galleryModel) {
    'use strict';

    var splideVideo = galleryModel.splideVideo,
        splideSlider = galleryModel.splideSlider,
        TOP = 'top',
        RIGHT = 'right',
        BOTTOM = 'bottom',
        LEFT = 'left',

        /**
         * @constructor
         *
         * @param {Object} context
         */
        SplideSliderHandler = function (context) {
            this.context = context;
        };

    // constructor -> add methods
    SplideSliderHandler.prototype = {
        /**
         * Init splide slider and thumbnails for custom gallery
         *
         * @return {void}
         */
        initSplide: function () {
            var self = this,
                context = this.context;

            try {
                if ($(context.galleryContainerId).length) {
                    context.mainSplide = new splideSlider(context.galleryContainerId, self.getSliderOptions());
                    self.initSplideListeners();
                    self.refreshSliderByItemsCount();
                }

                if (context.thumbnailsEnabled) {
                    context.mainSplide.options.index = self.getMainImageSlideIndex();
                    self.initThumbnails();
                    self.initResponsive();
                } else {
                    context.mainSplide.mount({ splideVideo });
                }
            } catch (err) {
                console.log(err);
            }
        },

        /**
         * Get index for the main image
         * depending on the configured image role
         *
         * @return {number}
         */
        getMainImageSlideIndex: function () {
            var index = 0,
                context = this.context;

            if (context.imagesList) {
                _.each(context.imagesList, function (image, i) {
                    if (image.isMain) {
                        index = i;
                    }
                })
            }

            index = context.mainSliderOptions.start
                ? context.mainSliderOptions.start : index;

            return index;
        },

        /**
         * Toggle class for slider wrapper depending on quantity of sliders
         * hide arrows and pagination if 1 image
         * @param {Object} imageData
         *
         * @return {void}
         */
        refreshSliderByItemsCount: function (imageData) {
            var context = this.context,
                images = imageData ? imageData : context.imagesList;

            if (!images) {
                images = context.context.imagesList;
            }

            if (images && !(images.length > 1)) {
                context.galleryWrapper.addClass(context.classes.oneItemClass);
            } else {
                context.galleryWrapper.removeClass(context.classes.oneItemClass);
            }
        },

        /**
         * Init splide slider actions for custom gallery
         *
         * @return {void}
         */
        initSplideListeners: function () {
            var context = this.context,
                activeIndex,
                sliderArrows = $(context.galleryContainerId).find(context.selectors.sliderArrows);

            // hide preview image when slider is ready
            context.mainSplide.on('mounted', function () {
                if (context.preloadImage && context.preloadImage.length) {
                    context.preloadImage.css('padding-bottom', 0);

                    $(context.mainSplide.Components.Elements.list)
                        .find('img').eq(context.mainSplide.index).one('load', function() {
                        context.preloadImage.hide();
                    });
                }
            });

            if (context.zoomConfig.isZoomEnabled && !galleryModel.isTouchDevice()) {
                context.mainSplide.on('mounted.zoom', function () {
                    // fix for fullscreen initial slider (will take the index from the
                    // main slider in context
                    activeIndex = context.context && context.context.mainSplide
                        ? context.context.mainSplide.index : context.mainSplide.index;

                    if (!activeIndex) {
                        activeIndex = context.mainSplide.index;
                    }

                    context.currentSlide = this.getSlideByIndex(activeIndex);
                    this.refreshZoom();
                }.bind(this));

                context.mainSplide.on('move.zoom', function (nextSlideIndex) {
                    context.currentSlide = this.getSlideByIndex(nextSlideIndex);
                    this.refreshZoom();
                }.bind(this));

                context.mainSplide.on('destroy', function () {
                    context.zoom.destroyZoom();
                });
            }

            if (!context.galleryWrapperWatcherEnabled && !galleryModel.isTouchDevice()) {
                context.galleryWrapper.on('gallery:zoom:refresh', function () {
                    context.currentSlide = this.getSlideByIndex(context.mainSplide.index);
                    this.refreshZoom();
                }.bind(this));

                context.galleryWrapperWatcherEnabled = true;
            }

            // make slider arrows shorter for video slides
            // to prevent iframe controls overlap with custom controls
            context.mainSplide.on('video:play', function () {
                sliderArrows.addClass(context.classes.shortClass);
            });

            context.mainSplide.on('video:pause', function () {
                sliderArrows.removeClass(context.classes.shortClass);
            });

            // set additional listeners from context
            // context.getAdditionalListeners - should be an array with additional listeners
            if (context.getAdditionalListeners && context.getAdditionalListeners.length) {
                _.each(context.getAdditionalListeners, function (eventListener) {
                    eventListener(context.mainSplide);
                })
            }
        },

        /**
         * Refresh zoom if it is enabled
         *
         * @return {void}
         */
        refreshZoom: function () {
            var context = this.context;

            if (_.has(context, 'zoom')) {
                context.zoom.destroyZoom();
                context.zoom.attachZoom(context);
            }
        },

        /**
         * Update gallery options depending on special conditions
         *
         * @return {object}
         */
        getSliderOptions: function () {
            var context = this.context,
                sliderOptions = context.mainSliderOptions,
                breakpoint = context.mediaBreakpoint - 1;

            sliderOptions.start = this.getMainImageSlideIndex();

            if (context.galleryWrapper.parents(context.selectors.popupWrapperContainer).length) {
                context.thumbnailsEnabled = false;
            }

            if (context.galleryConfig.lazyLoadEnabled) {
                sliderOptions.lazyLoad = 'nearby';
                sliderOptions.preloadPages = 1;
            }

            if (context.thumbnailsEnabled) {
                sliderOptions.pagination = false;
                sliderOptions.breakpoints = {
                    [breakpoint]: {
                        pagination: true
                    }
                };
            }

            return sliderOptions;
        },

        /**
         * Get current slide element by slide ID
         * @param index
         *
         * @returns {null}
         */
        getSlideByIndex: function (index) {
            var slideModel,
                slide = null;

            if (_.has(this.context, 'mainSplide')) {
                slideModel = this.context.mainSplide.Components.Elements.slides[index];

                if (slideModel) {
                    slide = $(slideModel);
                }

            }

            return slide
        },


        //
        //  Thumbnails
        //  --------------------------------------------

        /**
         * Set thumbnails options and clones template for future use
         * Mounting will be done by responsivity init: initResponsive()
         *
         * @return {void}
         */
        initThumbnails: function () {
            var self = this,
                context = this.context;

            self.setThumbsOptions();

            switch (context.galleryConfig.thumbnailsPosition) {
                case LEFT:
                case RIGHT:
                    context.thumbsVerticalDirection = true;
                    self.setVerticalThumbsOptions();

                    break;
                case TOP:
                case BOTTOM:
                    context.thumbsVerticalDirection = false;
                    self.setHorizontalThumbsOptions();

                    break;
            }

            context.thubmsTmp = self.getThumbsTemplateClone();
        },

        /**
         * Set common options for thumbnails
         *
         * @return {void}
         */
        setThumbsOptions: function () {
            _.extend(this.context.thumbnailOptions, {
                arrows: this.context.viewXmlOptions.arrows,
                gap: this.context.viewXmlOptions.thumbmargin,
                fixedWidth: this.context.viewXmlOptions.thumbwidth,
                fixedHeight: this.context.viewXmlOptions.thumbheight
            });
        },

        /**
         * Set special options for vertical thumbnails
         *
         * @return {void}
         */
        setVerticalThumbsOptions: function () {
            _.extend(this.context.thumbnailOptions, {
                focus: 0,
                direction: 'ttb',
                height: this.getVerticalSlidesHeight(),
                width: this.context.viewXmlOptions.thumbwidth,
                fixedWidth: this.context.viewXmlOptions.thumbwidth
            });
        },

        /**
         * Set special options for horizontal thumbnails
         *
         * @return {void}
         */
        setHorizontalThumbsOptions: function () {
            _.extend(this.context.thumbnailOptions, {
                autoWidth: true,
                direction: 'ltr',
                height: this.context.viewXmlOptions.thumbheight,
                fixedHeight: this.context.viewXmlOptions.thumbheight + 2
            });
        },

        /**
         * Mount and sync thumbnails with main gallery
         *
         * @param {boolean} FORCE
         *
         * @return {void}
         */
        mountThumbnails: function (FORCE) {
            var self = this,
                context = this.context,
                primarySlider = context.mainSplide,
                thumbnailsId = context.galleryThumbnailsId,
                thumbnailsOptions = context.thumbnailOptions;

            if ($(thumbnailsId).length && !context.thumbnailsSplide || FORCE) {
                context.thumbnailsSplide = new splideSlider(thumbnailsId, thumbnailsOptions);
                self.initThumbsListeners(thumbnailsId);
                context.thumbnailsSplide.mount();
            }

            // if main slider still not mounted it can be just mount
            // otherwise it should be destroyed first to reset all components and listeners
            if (primarySlider.state.is(splideSlider.STATES.CREATED)
                && !primarySlider.state.is(splideSlider.STATES.MOUNTED)) {
                primarySlider.sync(context.thumbnailsSplide).mount({ splideVideo });
            } else {
                primarySlider.destroy(false);
                this.initSplideListeners();
                primarySlider.options.pagination = false;
                primarySlider.sync(context.thumbnailsSplide).mount({ splideVideo });
            }

            if (context.thumbsVerticalDirection) {
                this.setVerticalSlidesHeight();
            } else {
                this.setHorizontalSlidesWidth();
            }
        },

        /**
         * Init thumbnails slider actions
         *
         * @param {string} sliderId
         *
         * @return {void}
         */
        initThumbsListeners: function (sliderId) {
            var context = this.context,
                target,
                targetSlide,
                delayedHoverEvent = _.throttle(function (e) {
                    target = $(e.target).hasClass(context.selectors.imageContainer)
                        ? $(e.target) : $(e.target).parents('.' + context.selectors.imageContainer).first();
                    targetSlide = target.attr('data-slide');
                    if (targetSlide && !target.hasClass(context.classes.activeClass)) {
                        context.thumbnailsSplide.go(+targetSlide);
                    }
                }, context.throttleDelay);

            // switch main image on thumbs hover
            if (context.galleryConfig.flipMainImageOnHover && $(sliderId).length) {
                $(sliderId).on('mouseenter mousemove', function (e) {
                    delayedHoverEvent(e);
                });
            }
        },

        /**
         * Clones current thumbnails template
         *
         * @return {*|jQuery}
         */
        getThumbsTemplateClone: function () {
            return $(this.context.galleryThumbnailsId).clone();
        },

        /**
         * Get correct number for height ration of thumbnails
         * to avoid big white spaces
         *
         * @return {number}
         */
        getVerticalSlidesHeight: function () {
            var preloadImage = this.context.galleryWrapper.find(this.context.selectors.preloadImage),
                maxAllowedHeight = preloadImage.height() - this.context.arrowsPadding * 2,
                thumbOuterHeight = this.context.thumbnailOptions.fixedHeight + this.context.thumbnailOptions.gap,
                perPegaImagesAllowed = Math.floor(maxAllowedHeight / thumbOuterHeight);

            if (!maxAllowedHeight || maxAllowedHeight > this.context.galleryWrapper.height()) {
                maxAllowedHeight = this.context.galleryWrapper.height() - this.context.arrowsPadding * 2;
            }

            if (this.context.imagesList && this.context.imagesList.length <= perPegaImagesAllowed) {
                maxAllowedHeight = this.context.imagesList.length * thumbOuterHeight - this.context.arrowsPadding / 2;
            }

            return maxAllowedHeight;
        },

        /**
         * Get correct width for thumbnails slider
         * to avoid big white spaces
         *
         * @return {Object}
         */
        getHorizontalSlidesWidth: function () {
            var slideBorderWidth = 1,
                thumbsWidth = this.context.thumbnailOptions.fixedWidth,
                thumbsGap = this.context.thumbnailOptions.gap,
                maxAllowedWidth = this.context.galleryWrapper.width(),
                thumbOuterWidth = thumbsWidth + thumbsGap + slideBorderWidth * 2,
                perPegaImagesAllowed = Math.floor(maxAllowedWidth / thumbOuterWidth);

            if (this.context.imagesList && this.context.imagesList.length <= perPegaImagesAllowed) {
                // eslint-disable-next-line max-len
                maxAllowedWidth = this.context.imagesList.length * thumbOuterWidth - thumbsGap + this.context.arrowsPadding * 2;
            }

            return maxAllowedWidth;
        },

        /**
         * Set correct number for height ration of thumbnails
         * to avoid big white spaces
         *
         * @return {Object}
         */
        setVerticalSlidesHeight: function () {
            this.context.thumbnailsSplide.options.height = this.getVerticalSlidesHeight();
            this.context.thumbnailsSplide.refresh();

            return this;
        },

        /**
         * Set correct width for thumbnails slider
         * to avoid big white spaces
         *
         * @return {Object}
         */
        setHorizontalSlidesWidth: function () {
            this.context.galleryWrapper.find(this.context.galleryThumbnailsId).css({
                'max-width': this.getHorizontalSlidesWidth(),
                'margin-right': 'auto',
                'margin-left': 'auto'
            });

            return this;
        },

        /**
         * Set listener for resize
         * note: will be used only for the main slider on desktop,
         * fullscreen uses another function because of a bit different templates
         *
         * @return {SplideSliderHandler}
         */
        initResizeListener: function () {
            var self = this;

            if (this.context.thumbnailsSplide
                && this.context.galleryConfig.thumbnailsPosition === LEFT
                || this.context.galleryConfig.thumbnailsPosition === RIGHT) {
                $(window).on('resize.galleryThumbnails', _.debounce(function () {
                    self.setVerticalSlidesHeight();
                }, self.context.throttleDelay));
            }

            if (this.context.thumbnailsSplide
                && this.context.galleryConfig.thumbnailsPosition === BOTTOM) {
                $(window).on('resize.galleryThumbnails', _.debounce(function () {
                    self.setHorizontalSlidesWidth();
                }, self.context.throttleDelay));

                this.context.galleryWrapper.on('gallery:swatches:updated', _.debounce(function () {
                    self.setHorizontalSlidesWidth();
                }, self.context.throttleDelay));
            }

            return this;
        },


        //
        //  Responsive rules
        //  --------------------------------------------

        /**
         * Will be fired only if thumbnails are enabled
         *
         * @return {void}
         */
        initResponsive: function () {
            var self = this,
                context = this.context;

            // listener need to be set only once per slider context
            // i.g. fullscreen slider has splideInit each time when fullscreen called
            // but this global listener should be fired only once
            if (!context.mediaWatcherActive) {
                mediaCheck({
                    media: '(min-width: ' + context.mediaBreakpoint + 'px)',

                    entry: function () {
                        self.toggleDesktopMode();
                    },
                    exit: function () {
                        self.toggleMobileMode();
                    }
                });

                if (context.enableResizeListener) {
                    self.initResizeListener();
                }

                context.mediaWatcherActive = true;
            } else {
                $(document).width() > context.mediaBreakpoint
                ? self.toggleDesktopMode()
                : self.toggleMobileMode();
            }
        },

        /**
         * Init thumbnails for desktop view
         *
         * @return {void}
         */
        toggleDesktopMode: function () {
            var self = this,
                context = this.context,
                primarySlider = context.mainSplide,
                secondarySlider = context.thumbnailsSplide,
                activeIndex = primarySlider && primarySlider.options.index ? primarySlider.options.index : 0;

            // if no thumbnails -> append them to the DOM
            if (!$(context.galleryThumbnailsId).length) {
                // if ajax popup works - we get additional thumbs carousel in main slider
                // even if thumbs are disabled for ajax popups by default
                // todo: find why and fix in another way
                context.galleryWrapper.find("[id$=" + context.classes.thumbnails + "]").first().remove();
                context.galleryWrapper.append(context.thubmsTmp);
            }

            // if thumbnails are destroyed -> mount them and sync with the main slider
            if (!secondarySlider || secondarySlider.state.is(splideSlider.STATES.DESTROYED)) {
                self.mountThumbnails(true);
            }

            if (context.newThumbsTmplCreated) {
                context.thumbnailsSplide.go(activeIndex);

                return;
            }

            // if image data was updated in mobile mode ->
            // -> force update data for thumbs slider by desktop mode toggle
            if (context.thumbnailsSplide && context.imageListBySwatchUpdated) {
                context.updateData({}, true);
                context.imageListBySwatchUpdated = false;
            } else {
                context.thumbnailsSplide.go(activeIndex);
            }
        },

        /**
         * Init thumbnails for mobile view
         * Destroy thumbnails to show dots as pagination
         *
         * @return {void}
         */
        toggleMobileMode: function () {
            var self = this,
                context = this.context,
                primarySlider = context.mainSplide,
                secondarySlider = context.thumbnailsSplide;

            // if page loads in mobile mode and not initialized as slider -> mount it
            if (primarySlider && primarySlider.state.is(splideSlider.STATES.CREATED)
                && !primarySlider.state.is(splideSlider.STATES.MOUNTED)) {
                primarySlider.mount({ splideVideo });
            }

            // if thumbs-slider exist -> destroy it
            if (secondarySlider) {
                secondarySlider.destroy();
            }

            // save current thumbs template for future reinit and remove it from the DOM
            context.thubmsTmp = self.getThumbsTemplateClone();
            $(context.galleryThumbnailsId).remove();
        }
    };

    return SplideSliderHandler;
});
