/**
 * Custom gallery - zoom
 */

define([
    'ko',
    'jquery',
    'underscore',
    'matchMedia',
    'Amasty_Gallery/js/model/am-gallery-model',
    'elevateZoom'
], function (ko, $, _, mediaCheck, galleryModel) {
    'use strict';

    return {
        context: {},
        galleryZoomConfig: {
            constrainType: 'height',
            constrainSize: 150,
            zoomType: 'lens',
            borderSize: 1,
            containLensZoom: true,
            cursor: 'pointer',
            imageCrossfade: true,
            lensShape: 'round',
            lensSize: 150,
            zoomActivation: 'hover',
            responsive: true
        },
        mediaBreakpoint: '(min-width: 1024px)',
        classes: {
            zoomEnabled: 'zoom-enabled'
        },
        currentImage: null,
        isMobile: false,
        prevElementPosition: 'static',

        /**
         * Initialize zoom
         *
         * @param {Object} context
         *
         * @return {void}
         */
        init: function (context) {
            this.context = context;

            if (!galleryModel.isTouchDevice()) {
                this._updateConfig();
                this._initResponsive();
            }
        },

        /**
         * Handle zoom on mediaBreakpoint
         */
        _initResponsive: function () {
            var self = this;

            mediaCheck({
                media: self.mediaBreakpoint,
                entry: function () {
                    self.isMobile = false;
                },
                exit: function () {
                    self.isMobile = true;

                    self.destroyZoom();

                    if (self.galleryZoomConfig.zoomActivation === 'click' && self.currentImage) {
                        self.currentImage.off('click', this._zoomOnClickCallback);
                    }
                }
            });
        },

        /**
         * Get image elements from block
         *
         * @param {Element} block
         * @return {Element|null}
         */
        retrieveImage: function (block) {
            var image = null;

            if (block && block.find('img').data('zoom-image')) {
                image = block.find('img');
            }

            return image;
        },

        /**
         * Attach zoom to current gallery image
         *
         * @param {Object} context
         * @return {void}
         */
        attachZoom: function (context) {
            this.currentImage = this.retrieveImage(context.currentSlide);

            if (!this.currentImage || this.isMobile) {
                return;
            }

            if (!this.currentImage.attr('src')) {
                this.currentImage.on('load', function () {
                    this.initZoomProcess();
                }.bind(this));
            } else {
                this.initZoomProcess();
            }
        },

        /**
         *  Attach zoom to current gallery image
         *
         * @return {void}
         */
        initZoomProcess: function () {
            if (this.elevateZoom) {
                this.destroyZoom();
            }

            if (this.galleryZoomConfig.zoomActivation === 'click') {
                this.zoomOnClick();

                return;
            }

            this.prevElementPosition = this.currentImage.css('position');
            this.elevateZoom = this.currentImage.elevateZoom(this.galleryZoomConfig);
            this.isZoomInitialized = true;
        },

        /**
         * Destroy elevate zoom on image
         *
         * @return {void}
         */
        destroyZoom: function () {
            if (!this.elevateZoom) {
                return;
            }

            $('.zoomContainer').remove();
            this.elevateZoom.removeData('elevateZoom');
            this.elevateZoom.removeData('zoomImage');

            if (this.elevateZoom.parent().hasClass('zoomWrapper')) {
                this.elevateZoom.unwrap().css('position', this.prevElementPosition);
            }

            this.isZoomInitialized = false;
        },

        /**
         * Zoom handler on image click
         *
         * @private
         * @param {Event} event
         * @return {void}
         */
        _zoomOnClickCallback: function (event) {
            if (this.currentImage.hasClass(this.classes.zoomEnabled)) {
                this.currentImage.removeClass(this.classes.zoomEnabled);
                this.destroyZoom();

                return true;
            }

            event.preventDefault();
            event.stopImmediatePropagation();

            this.prevElementPosition = this.currentImage.css('position');
            this.elevateZoom = this.currentImage.elevateZoom(this.galleryZoomConfig);
            this.currentImage.addClass(this.classes.zoomEnabled);
        },

        /**
         * Enable elevate zoom on image click
         *
         * @return {void}
         */
        zoomOnClick: function () {
            this.currentImage.off('click').on('click', this._zoomOnClickCallback.bind(this));
        },

        /**
         * Update zoom config
         *
         * @private
         * @return {void}
         */
        _updateConfig: function () {
            if (_.has(this.context, 'zoomConfig')) {
                $.extend(this.galleryZoomConfig, this.context.zoomConfig);
            }
        }
    }
});
