/**
 * Amasty custom gallery for product pages
 */

define([
    'jquery',
    'underscore',
    'uiClass',
    'matchMedia',
    'mage/template',
    'Amasty_Gallery/js/am-zoom',
    'Amasty_Gallery/js/am-video',
    'Amasty_Gallery/js/am-fullscreen',
    'Amasty_Gallery/js/model/sliders-model',
    'Amasty_Gallery/js/model/am-gallery-model',
    'text!Amasty_Gallery/template/am-gallery.html',
    'text!Amasty_Gallery/template/am-gallery-item.html',
    'text!Amasty_Gallery/template/am-gallery-thumbnails.html'
], function ($, _, Class, mediaCheck, template, zoom, Video, Fs, SlidersModel, galleryModel,
             galleryTpl, galleryItemTpl, galleryThumbsTpl) {
    'use strict';

    var splideVideo = galleryModel.splideVideo,
        splideSlider = galleryModel.splideSlider;

    return Class.extend({
        defaults: {
            imagesList: {}, // object with images data
            productName: '', // product name for 'alt' attr on images in the gallery
            viewXmlOptions: {}, // options from the view.xml - need for thumbs size and gap
            throttleDelay: 200, // throttle on mousemove - need for switching images on thumbs hover
            verticalHeightRatio: 2, // height ration for vertical thumbnails on desktop
            arrowsPadding: 32,
            galleryConfig: { // custom gallery configs (custom options)
                useOptimizedImages: true, // if false - not resized by Magento from view.xml sizes
                isFullscreenModeEnabled: true,
                lazyLoadEnabled: true, // if true - lazy load from splide-gallery works
                slidersGap: 20, // padding between thumbs and main slider - for left or right thumbs position only
                thumbnailsEnabled: true, // if false - dots as pagination will be used even for desktop
                thumbnailsPosition: 'left',
                flipMainImageOnHover: false // if true - main gallery images changed on thumbnails hover
            },
            selectors: {
                galleryId: 'am-gallery',
                gallerySelector: '#am-gallery',
                thumbnailsId: 'am-gallery-thumbnails',
                thumbnailsSelector: '#am-gallery-thumbnails',
                sliderArrows: '.splide__arrow',
                imageContainer: 'splide__slide',
                sliderContainer: '.splide__slider',
                preloadImage: '.gallery-placeholder__image',
                videoSlide: '.splide__slide .video-slide',
                videoPlayButton: '[data-role="video-play-button"]',
                popupWrapperContainer:'.amtheme-popup-block'
            },
            classes: {
                prevClass: '--prev',
                nextClass: '--next',
                shortClass: '-short',
                processed: '-processed',
                thumbnails: '-thumbnails',
                activeClass: 'is-active',
                fullscreenClass: 'fs-active',
                oneItemClass: '-one-slide'
            },
            mainSliderOptions: { // main gallery config (native slider options)
                perPage: 1,
                type: 'fade',
                rewind: true,
                video: {
                    loop: true
                }
            },
            thumbnailOptions: { // thumbnails config (native slider options)
                rewind: true,
                pagination: false,
                isNavigation: true,
                updateOnMove: true
            },
            zoomConfig: {
                isZoomEnabled: false,
                isFsZoomEnabled: false,
                zoomActivation: 'hover'
            }
        },

        /**
         * @inheritDoc
         *
         * @param {object} config
         * @param {dom} element
         *
         * @return {void}
         */
        initialize: function (config, element) {
            this._super();

            // widget main variables
            this._initVariables(element);

            // gallery and thumbnails template
            this._buildTemplate();

            // init zoom
            this.initZoom();

            // init slider and thumbnails
            this._initSplide();

            // init swatches data and actions
            this._setGalleryDataAndActions();

            // init fullscreen
            this._initFullscreen();
        },

        /**
         * Init zoom if it is enabled
         *
         * @return {Object}
         */
        initZoom: function () {
            if (this.zoomConfig.isZoomEnabled) {
                this.zoom = zoom;
                this.zoom.init(this);
            }

            return this;
        },

        /**
         * Set scope variables
         *
         * @param {dom} element
         *
         * @return {void}
         */
        _initVariables: function (element) {
            // flag -> will be used by swatches update
            this.imageListBySwatchUpdated = false;

            // flag -> will be used to change thumbs heightratio if thumbs enabled
            this.enableResizeListener = true;

            // dom elements
            this.galleryWrapper = $(element);
            this.preloadImage = this.galleryWrapper.find(this.selectors.preloadImage);

            // scope variables
            this.galleryContainerId = this.selectors.gallerySelector;
            this.galleryThumbnailsId = this.selectors.thumbnailsSelector;
            this.thumbnailsEnabled = this.galleryConfig.thumbnailsEnabled;
            this.mediaBreakpoint = this.breakpoints.desktopBreakpoint;
            this.useOptimizedImages = this.galleryConfig.useOptimizedImages;

            // update default images list -> add video type to the items with video
            this.imagesList = Video.modifyDataList(this.imagesList);
        },

        /**
         * Init fullscreen if it is enabled
         *
         * @return {Class}
         */
        _initFullscreen: function () {
            if (this.galleryConfig.isFullscreenModeEnabled) {
                this.galleryWrapper.addClass(this.classes.fullscreenClass);

                Fs.viewXmlOptions = this.viewXmlOptions;
                Fs.galleryConfig = this.fullscreenConfig;
                Fs.zoomConfig = this.zoomConfig;
                Fs.mediaBreakpoint = this.breakpoints.fullscreenBreakpoint;
                Fs.fsInit(this);
            }

            return this;
        },

        /**
         * Fill the template with the data using underscore template notation,
         * add template to the DOM
         *
         * @return {Class}
         */
        _buildTemplate: function () {
            if (this.imagesList.length < 2) {
                this.thumbnailsEnabled = false;
            }

            this.tmpl = template(galleryTpl, {
                data: this.imagesList,
                productName: this.productName,
                mainId: this.selectors.galleryId,
                thumbsId: this.selectors.thumbnailsId,
                thumbnailsEnabled: this.thumbnailsEnabled,
                useOptimizedImages: this.useOptimizedImages,
                mainImgHeight : '100%', // can't be set an actual size because of M-2.3.7 bug
                mainImgWidth: this.viewXmlOptions.width || '100%',
                thumbImgHeight: this.viewXmlOptions.thumbwidth || '100%',
                thumbImgWidth: this.viewXmlOptions.thumbwidth || '100%',
                lazyLoadEnabled: this.galleryConfig.lazyLoadEnabled
            });

            this.galleryWrapper.append(this.tmpl.trim());

            // jquery object for gallery appended template
            this.galleryContainer = this.galleryWrapper.find(this.galleryContainerId);

            return this;
        },


        //
        //  Main gallery
        //  --------------------------------------------

        /**
         * Init splide slider for custom gallery with current scope options
         * Init splide listeners, init gallery thumbnails, apply gallery options
         *
         * @return {void}
         */
        _initSplide: function () {
            this._updateConfig();
            this.sliderModel = new SlidersModel(this);
            this.sliderModel.initSplide();
        },

        /**
         * Update config from window.additionalConfig object
         * This config help us to update slider options in different modes
         * @private
         * @return {void}
         */
        _updateConfig: function () {
            if(_.has(window, 'gallery') && _.has(window.gallery, 'additionalConfig')) {
                $.extend(true, this, window.gallery.additionalConfig);
            }
        },

        //
        //  Swatches
        //  --------------------------------------------

        /**
         * This function is required for gallery update on swatches
         * Add data to the dom element and add  methods to the am-gallery api
         *
         * @return {void}
         */
        _setGalleryDataAndActions: function () {
            this.returnCurrentImages = this._returnCurrentImages;
            this.updateData = this._updateData;

            this.galleryWrapper.data('gallery', this);
            this.galleryWrapper.trigger('gallery:loaded');
        },

        /**
         * Swatch-renderer.js triggers this function on swatches click
         *
         * @private
         *
         * @param {object} swatchData
         * @param {boolean} FORCE - if true -> updates (refresh) data even without swatches click
         *
         * @return {object || void}
         */
        _updateData: function (swatchData, FORCE) {
            var self = this,
                activeIndex = 0,
                secondarySlider,
                primarySlider = this.mainSplide;

            this.newThumbsTmplCreated = false;
            swatchData = FORCE ? this.imagesList : Video.modifyDataList(swatchData);

            if (!swatchData.length || !swatchData[0].hasOwnProperty('type')) {
                return this.imagesList;
            } else if (FORCE || !_.isEqual(this.imagesList, swatchData)) {
                this.imagesList = swatchData;
                this.setProcessState();
                this.clearSlides();

                if (!this.galleryWrapper.parents(this.selectors.popupWrapperContainer).length) {
                    this.oneImageToggle(swatchData);
                }

                this.sliderModel.refreshSliderByItemsCount(swatchData);
                this.buildSlides(swatchData);

                secondarySlider = this.thumbnailsSplide;

                // set actual active index if update goes by media switch
                // and active slide shouldn't be 0
                if (FORCE) {
                    activeIndex = primarySlider.index;
                }

                if (secondarySlider && !secondarySlider.state.is(splideSlider.STATES.DESTROYED)) {
                    // update action on desktop and for thumbs
                    self.sliderModel.mountThumbnails(false);
                    secondarySlider.go(activeIndex);
                    primarySlider.go(activeIndex);

                    // save new thumbs template in scope -> use it by media (desktop/mobile) switch
                    self.thubmsTmp = self.sliderModel.getThumbsTemplateClone();
                } else {
                    // update action on mobile or if no thumbs enabled
                    primarySlider.destroy();
                    self.sliderModel.initSplideListeners();
                    primarySlider.mount({ splideVideo }).go(activeIndex);
                }

                this.resetProcessState();
                this.imageListBySwatchUpdated = true;
            }
        },

        /**
         * @return {Object}
         */
        setProcessState: function () {
            var currentWrapperHeight = this.galleryWrapper.outerHeight();

            this.galleryWrapper.addClass(this.classes.processed).css('height', currentWrapperHeight);

            return this;
        },

        /**
         * @return {Object}
         */
        resetProcessState: function () {
            var self = this;

            // hide loader only after first image is fully loaded and takes its place
            $(this.mainSplide.Components.Elements.list).find('img').first().on('load', function () {
                _.delay(function () {
                    self.galleryWrapper.removeClass(self.classes.processed).css('height', '');

                    if (self.zoomConfig.isFsZoomEnabled) {
                        self.galleryWrapper.trigger('gallery:zoom:refresh');
                    }
                }, self.throttleDelay)
            });

            self.galleryWrapper.trigger('gallery:swatches:updated');

            return this;
        },

        /**
         * Clear current slides - remove all slides
         * from gallery and thumbnails if they are enabled
         *
         * @return {Object}
         */
        clearSlides: function () {
            var primarySlider = this.mainSplide,
                secondarySlider = this.thumbnailsSplide;

            $(primarySlider.Components.Elements.list).empty();

            if (secondarySlider) {
                $(secondarySlider.Components.Elements.list).empty();
            }

            return this;
        },

        /**
         * Build and add slides with new images
         * for main gallery and thumbnails if they are enabled
         * @param {Object} swatchData
         *
         * @return {Object}
         */
        buildSlides: function (swatchData) {
            var self = this,
                thumbsTmpl,
                galleryTmpl,
                dataHasVideo = false,
                primarySlider = this.mainSplide,
                secondarySlider = this.thumbnailsSplide;

            _.each(swatchData, function (item, index) {
                galleryTmpl = template(galleryItemTpl, {
                    dataType: 'main',
                    dataItem: item,
                    dataSlide: index,
                    productName: self.productName,
                    useOptimizedImages: self.useOptimizedImages
                });

                primarySlider.add(galleryTmpl.trim());

                if (secondarySlider && !self.newThumbsTmplCreated) {
                    thumbsTmpl = template(galleryItemTpl, {
                        dataType: 'thumb',
                        dataItem: item,
                        dataSlide: index,
                        productName: self.productName,
                        useOptimizedImages: self.useOptimizedImages
                    });

                    secondarySlider.add(thumbsTmpl.trim());
                }

                if (item.type === 'video') {
                    dataHasVideo = true;
                }

                if (dataHasVideo || self.galleryConfig.lazyLoadEnabled) {
                    primarySlider.options.pagination = !secondarySlider;
                }
            });

            return this;
        },

        /**
         * Show / hide thumbnails depending on images count
         * @param {Object} swatchData
         *
         * @return {Object}
         */
        oneImageToggle: function (swatchData) {
            var secondarySlider = this.thumbnailsSplide;

            // add and init thumbnails if config-product after swatch click gets more than 1 images
            if (this.galleryConfig.thumbnailsEnabled && !secondarySlider && swatchData.length > 1) {
                this.galleryWrapper.append(
                    template(galleryThumbsTpl, {
                        data: swatchData,
                        productName: this.productName,
                        thumbsId: this.selectors.thumbnailsId,
                        useOptimizedImages: this.useOptimizedImages,
                        lazyLoadEnabled: false
                    })
                );
                this.newThumbsTmplCreated = true;

                this.sliderModel.initThumbnails();
                this.sliderModel.initResponsive();
            }

            // if after swatch click a product has only 1 image - remove secondary slider fully from
            // the DOM, remove padding for main slider
            if (this.galleryConfig.thumbnailsEnabled && secondarySlider && swatchData.length === 1) {
                secondarySlider.destroy();
                $(this.selectors.thumbnailsSelector).remove();
                this.thumbnailsSplide = null;
            }

            return this;
        },

        /**
         * Function for the swatch-renderer.js
         *
         * @private
         *
         * @return {object}
         */
        _returnCurrentImages: function () {
            return this.imagesList;
        }
    })
})
