/**
 * Custom gallery - fullscreen
 */

define([
    'ko',
    'jquery',
    'underscore',
    'mage/translate',
    'mage/template',
    'fastAverageColor',
    'Amasty_Gallery/js/am-zoom',
    'Amasty_Gallery/js/model/sliders-model',
    'Amasty_Gallery/js/model/am-gallery-model',
    'text!Amasty_Gallery/template/am-gallery.html'
], function (ko, $, _, $t, template, FastAverageColor, zoom, SlidersModel, galleryModel, galleryTpl) {
    'use strict';

    var BOTTOM = 'bottom',
        RIGHT = 'right',
        LEFT = 'left',
        ESC_CODE = 27,
        splideSlider = galleryModel.splideSlider,
        isFsActive = galleryModel.isFsActive,
        isFsProcess = galleryModel.isFsProcess;

    return {
        context: {}, // all data from custom-gallery
        throttleDelay: 200, // throttle on mousemove - need for switching images on thumbs hover
        galleryConfig: {}, // object comes from the main slider,
        arrowsPadding: 32,
        thumbsPerPage: 3,
        mainSliderOptions: { // options for fullscreen view
            type: 'slider',
            rewind: true,
            perPage: 1,
            video: {
                loop: true
            }
        },
        thumbnailOptions: { // thumbnails config (native slider options)
            focus: 0,
            perPage: 1,
            perMove: 1,
            rewind: true,
            pagination: false,
            isNavigation: true,
            updateOnMove: true
        },
        animations: {
            openAnimationsName: 'conditionalOpen',
            closeAnimationsName: 'conditionalClose'
        },
        classes: {
            fsProcess: 'gallery-fullscreen-process',
            fsActive: 'gallery-fullscreen-active',
            fsCloseBtn: 'gallery-fullscreen-close',
            fsPlusBtn: 'gallery-fullscreen-plus',
            fsMinusBtn: 'gallery-fullscreen-minus',
            fsBackgroundBlur: '-blurred',
            fsClose: 'to-close',
            fsOpen: 'to-open'
        },
        selectors: {
            galleryId: '',
            thumbnailsId: '',
            fsContainer: '-container',
            fsWrapper: '-wrapper',
            fsDefiner: '-fullscreen',
            fsSlider: '.splide__slider',
            videoSlideSelector: '[data-video="true"]'
        },

        /**
         * @param {object} context - main gallery Class
         *
         * @return {void}
         */
        fsInit: function (context) {
            this.sliderModel = new SlidersModel(this);

            this.setContext(context);
            this.setVariables();
            this.initFsListeners();
            this.initZoom();
        },

        /**
         * Init zoom if it is enabled
         *
         * @return {Object}
         */
        initZoom: function () {
            if (this.context.zoomConfig.isFsZoomEnabled) {
                this.zoom = zoom;
                this.zoom.init(this);
            }

            return this;
        },

        /**
         * Set defaults for fullscreen / extend some options from the context
         *
         * @return {void}
         */
        setVariables: function () {
            // scope variables
            this.getAdditionalListeners = [];
            this.swipeOnWheelScroll = this.galleryConfig.swipeOnWheelScroll;
            this.thumbnailsEnabled = this.galleryConfig.thumbnailsEnabled;

            // extend scope selectors
            this.classes = $.extend({}, this.context.classes, this.classes, true);
            this.selectors = $.extend({}, this.context.selectors, this.selectors, true);

            this.selectors.galleryId = this.context.selectors.galleryId + this.selectors.fsDefiner;
            this.selectors.thumbnailsId = this.context.selectors.thumbnailsId + this.selectors.fsDefiner;

            this.galleryContainerId = '#' + this.selectors.galleryId;
            this.galleryThumbnailsId = '#' + this.selectors.thumbnailsId;

            // build additional scope selectors variables
            this.fullscreenMainContainer = this.fullscreenMainContainer
                ? this.fullscreenMainContainer
                : this.context.selectors.galleryId + this.selectors.fsDefiner + this.selectors.fsContainer;

            this.fullscreenSlidersWrapper = this.fullscreenSlidersWrapper
                ? this.fullscreenSlidersWrapper
                : this.context.selectors.galleryId + this.selectors.fsDefiner + this.selectors.fsWrapper;

            this.thumbsVerticalDirection = this.galleryConfig.thumbnailsPosition === LEFT
                || this.galleryConfig.thumbnailsPosition === RIGHT;

            this.backgroundSettings = {
                type: this.galleryConfig.backgroundType || 'color',
                adaptive: this.galleryConfig.adaptiveBg || false,
                transparent: +this.galleryConfig.transparentBg || false,
                isBlurred: +this.galleryConfig.backgroundBlur || false,
                opacity: +this.galleryConfig.backgroundOpacity || 100,
                color: this.galleryConfig.backgroundColor || '#666666'
            };

            if (this.backgroundSettings.transparent) {
                this.backgroundSettings.isBlurred = 1;
            }
        },

        /**
         * @return {void}
         */
        initFsListeners: function () {
            var self = this,
                body = $('body'),
                context = this.context,
                cursorTarget,
                clientX,
                clientY;

            // open fullscreen on main slider item click
            // except videos -> they should start to play
            $(context.galleryContainer).on('click', 'LI', function () {
                var $this = $(this);

                if (!$this.is(self.selectors.videoSlideSelector)
                    && $this.hasClass(context.selectors.imageContainer)
                    && !isFsProcess() && !isFsActive()) {
                    self.fsOpen();
                }
            });

            // if fullscreen is active -> close it on ESC tap
            $(document).on('keyup', function (e) {
                if (e.key === 'Escape' || e.which === ESC_CODE) {
                    self.fsClose();
                }
            });

            // if thumbs vertical on fullscreen -> need to fix its height on resize
            // should fix thumbs scrolling when it's not required
            if (this.galleryConfig.thumbnailsEnabled && this.thumbsVerticalDirection) {
                $(window).on('resize', _.debounce(self.fixThumbsHeightOnResize, self.throttleDelay).bind(self));
            }

            // toggle actions on fs active state
            isFsActive.subscribe(function (data) {
                if (data) {
                    body.addClass(self.classes.fsActive);

                    if (self.backgroundSettings.isBlurred) {
                        body.addClass(self.classes.fsBackgroundBlur);
                    }
                } else {
                    body.removeClass(self.classes.fsActive);

                    if (self.backgroundSettings.isBlurred) {
                        body.removeClass(self.classes.fsBackgroundBlur);
                    }
                }
            });

            // toggle actions on fs process state
            isFsProcess.subscribe(function (data) {
                if (data) {
                    body.addClass(self.classes.fsProcess);
                } else {
                    body.removeClass(self.classes.fsProcess);
                }
            });

            // add listener to the main fullscreen slider to change fs background
            // while images changing
            if (this.backgroundSettings.adaptive) {
                this.getAdditionalListeners.push(self._setAdaptiveBackground.bind(self));
            }

            // set wheel on body to be able scroll images when the pointer over zoom container
            if (this.swipeOnWheelScroll && this.zoomConfig.isZoomEnabled) {
                this.setWheelListener(body);
            }

            // don't loose pointer focus on arrow by zoom change
            if (this.zoomConfig.isZoomEnabled) {
                body.on('click', function (e) {
                    if ($(e.target).hasClass('zoomContainer')
                        && e.clientX === clientX
                        && e.clientY === clientY) {
                            $(cursorTarget).trigger('click');
                        } else {
                            cursorTarget = e.target;
                        }

                    clientX = e.clientX;
                    clientY = e.clientY;
                })
            }
        },

        /**
         * Set context (variables and methods from the parent Class)
         *
         * @param {object} context
         *
         * @return {object}
         */
        setContext: function (context) {
            // eslint-disable-next-line no-return-assign
            return this.context = context;
        },

        /**
         * @return {*|boolean}
         */
        isThumbsVisible: function () {
            return this.thumbnailsEnabled
                && this.thumbnailsSplide
                && !this.thumbnailsSplide.state.is(splideSlider.STATES.DESTROYED);
        },

        /**
         * Create fs template, init slider and open it
         *
         * @return {void}
         */
        fsOpen: function () {
            var self = this,
                activeIndex = this.context.mainSplide.index;

            this.mainSliderOptions.start = activeIndex;
            this.thumbnailOptions.start = activeIndex;

            isFsProcess(true);

            try {
                this.buildTemplate();
                this.sliderModel.initSplide();
                this.setFsOpenActions(activeIndex);
                self.galleryWrapper.trigger('gallery:fs:opened');

                if (!this.galleryConfig.animationEnabled) {
                    isFsProcess(false);
                    isFsActive(true);

                    return;
                }

                this.fsSlidersContainer.on('animationend', function (event) {
                    if (event.originalEvent.animationName === self.animations.openAnimationsName) {
                        self.fsSlidersContainer.removeClass(self.classes.fsOpen);
                        isFsProcess(false);
                        isFsActive(true);
                    }
                });

                // fires open animation
                setTimeout(function () {
                    self.fsSlidersContainer.addClass(self.classes.fsOpen);
                }, 100);
            } catch (err) {
                console.log(err);

                if (this.fsSlidersContainer) {
                    this.fsSlidersContainer.remove();
                }

                isFsProcess(false);
            }
        },

        /**
         * @param {int} activeIndex
         *
         * @return {exports}
         */
        setFsOpenActions: function (activeIndex) {
            // for vertical thumbnails
            if (this.thumbnailsSplide && this.thumbsVerticalDirection) {
                this.fixThumbsHeightOnResize();
            }

            // for horizontal thumbnails
            if (this.thumbnailsEnabled && !this.thumbsVerticalDirection) {
                this.setMainGalleryNormalHeight();
                this.setThumbsContainerWidth();
            }

            if (this.thumbnailsSplide) {
                this.thumbnailsSplide.go(activeIndex, false);
            } else {
                this.mainSplide.go(activeIndex, false);
            }

            // if wheel scroll is enabled -> scroll main images on wheel
            if (this.swipeOnWheelScroll && !this.zoomConfig.isZoomEnabled) {
                this.setWheelListener(this.fsSlidersContainer);
            }

            return this;
        },

        /**
         * Close fullscreen
         *
         * @return {void}
         */
        fsClose: function () {
            var self = this,
                activeIndex = this.mainSplide.index,
                mainContext = this.context;

            isFsProcess(true);

            if (isFsActive() && this.fsSlidersContainer.length) {
                // open current image on the main desktop's slider
                if (mainContext.thumbnailsSplide) {
                    mainContext.thumbnailsSplide.go(activeIndex);
                } else {
                    mainContext.mainSplide.go(activeIndex);
                }
            }

            if (!this.galleryConfig.animationEnabled) {
                self.setFsCloseActions();

                return;
            }

            // if animation is enabled -> fullscreen window closes with scale to center
            this.fsSlidersContainer.on('animationend', function (event) {
                if (event.originalEvent.animationName === self.animations.closeAnimationsName) {
                    self.setFsCloseActions();
                }
            });

            // fires close animation
            this.fsSlidersContainer.addClass(this.classes.fsClose);
        },

        /**
         * @return {void}
         */
        setFsCloseActions: function () {
            this.mainSplide.destroy(true);

            if (this.thumbnailsSplide) {
                this.thumbnailsSplide.destroy(true);
            }

            if (this.context.zoomConfig.isFsZoomEnabled) {
                this.context.galleryWrapper.trigger('gallery:zoom:refresh');
            }

            this.fsSlidersContainer.remove();

            isFsProcess(false);
            isFsActive(false);
        },

        /**
         * @return {*|jQuery|HTMLElement}
         */
        buildCloseButton: function () {
            var self = this,
                buttonProps = {
                    'class': this.classes.fsCloseBtn,
                    'tabindex': 0,
                    'role': 'button',
                    'aria-label': $t('Exit fullscreen')
                },
                html = $('<div />', buttonProps);

            html.on('click', function (e) {
                e.preventDefault();

                self.fsClose();
            });

            return html;
        },

        /**
         * @return {*|jQuery|HTMLElement}
         */
        buildPlusMinusButtons: function (buttonCode) {
            var buttonName = 'fs' + buttonCode + 'Btn',
                buttonProps = {
                    'class': this.classes[buttonName],
                    'tabindex': 0,
                    'role': 'button'
                }

            return $('<div />', buttonProps);
        },

        /**
         * Build fs template and add it to the DOM
         *
         * @return {object}
         */
        buildTemplate: function () {
            var body = $('body'),
                // eslint-disable-next-line max-len
                containerClass = this.fullscreenMainContainer + ' -thumbs-position-' + this.galleryConfig.thumbnailsPosition,
                containerProps = {
                    'class': containerClass,
                    'id': this.fullscreenMainContainer,
                    'style': this.setFsBackground()
                },
                wrapperProps = {
                    'class': this.fullscreenSlidersWrapper,
                    'id': this.fullscreenSlidersWrapper
                };

            this.fsContainer = $('<div />', containerProps);
            this.fsWrapper = $('<div />', wrapperProps);

            if (this.context.imagesList.length < 2) {
                this.thumbnailsEnabled = false;
            }

            this.tmpl = template(galleryTpl, {
                useOptimizedImages: false,
                data: this.context.imagesList,
                mainId: this.selectors.galleryId,
                thumbsId: this.selectors.thumbnailsId,
                productName: this.context.productName,
                thumbnailsEnabled: this.thumbnailsEnabled,
                thumbImgHeight: this.context.viewXmlOptions.thumbwidth || '100%',
                thumbImgWidth: this.context.viewXmlOptions.thumbwidth || '100%',
                lazyLoadEnabled: this.galleryConfig.lazyLoadEnabled,
                mainImgWidth: '100%',
                mainImgHeight : '100%'
            });

            // wrap sliders and append to the DOM
            $(this.fsWrapper).wrapInner(this.tmpl.trim());
            body.append($(this.fsContainer).wrapInner(this.fsWrapper));

            // create close button and add to the fullscreen template
            $(this.fsContainer).prepend(this.buildCloseButton());
            $(this.fsContainer).prepend(this.buildPlusMinusButtons('Plus'));
            $(this.fsContainer).prepend(this.buildPlusMinusButtons('Minus'));

            this.galleryWrapper = body.find('#' + this.fullscreenSlidersWrapper);
            this.galleryContainer = body.find(this.galleryContainerId);
            this.fsSlidersContainer = body.find('#' + this.fullscreenMainContainer);

            return this;
        },

        /**
         * Swipe slides on mouse wheel scroll
         *
         * @param {Element} element
         *
         * @return {void}
         */
        setWheelListener: function (element) {
            var self = this,
                onWheel = _.throttle(function (e) {
                    if ($('body').hasClass(self.classes.fsActive)) {
                        var delta = e.originalEvent.deltaY || e.deltaY || e.detail || e.wheelDelta;

                        if (delta > 0) {
                            self.mainSplide.go('>');
                        } else {
                            self.mainSplide.go('<');
                        }
                    }
                }, this.throttleDelay);

            if ('onwheel' in document || 'onmousewheel' in document) {
                $(element).on('mousewheel', onWheel);
            } else {
                $(element).on('MozMousePixelScroll', onWheel);
            }
        },

        /**
         * Set fixed height for main slider container
         *
         * @return {this|object}
         */
        setMainGalleryNormalHeight: function () {
            var thumbItemHeight = this.thumbnailOptions.fixedHeight,
                thumbItemGap = this.thumbnailOptions.gap,
                extractedHeight;

            extractedHeight = thumbItemHeight + thumbItemGap;
            this.galleryContainer.css('max-height', 'calc(100% - ' + extractedHeight + 'px)');

            // eslint-disable-next-line no-unused-expressions
            this.galleryConfig.thumbnailsPosition === BOTTOM
                ? this.galleryContainer.css('marginBottom', thumbItemGap)
                : this.galleryContainer.css('marginTop', thumbItemGap);

            return this;
        },

        /**
         * Set fixed width for thumbnails container
         *
         * @return {this|object}
         */
        setThumbsContainerWidth: function () {
            var thumbOuterWidth = this.thumbnailOptions.fixedWidth + this.thumbnailOptions.gap,
                maxAllowedWidth,
                finalWidth,
                showSlides,
                perPage;

            if (this.isThumbnailsVisible() && this.galleryConfig.thumbnailsPosition === BOTTOM) {
                maxAllowedWidth = this.galleryWrapper.width();
                perPage = Math.floor(maxAllowedWidth / thumbOuterWidth);
                showSlides = perPage > this.thumbsPerPage ? this.thumbsPerPage : perPage;

                if (this.context.imagesList.length < showSlides) {
                    showSlides = this.context.imagesList.length;
                }

                finalWidth = showSlides * thumbOuterWidth + 20 * 2;

                $(this.galleryThumbnailsId).find(this.selectors.fsSlider).parent().width(finalWidth);

                this.thumbnailsSplide.options.width = finalWidth;
                this.thumbnailsSplide.refresh();
            }

            return this;
        },

        /**
         * Check if thumbs visible on full screen mode
         *
         * @return {boolean}
         */
        isThumbnailsVisible: function () {
            return (isFsActive() || isFsProcess()) && this.isThumbsVisible();
        },

        /**
         * Set fixed height for thumbnails container
         * Need to avoid big white spaces or slides overlapping
         *
         * @return {this|object}
         */
        fixThumbsHeightOnResize: function () {
            var thumbItemHeight = this.thumbnailOptions.fixedHeight,
                thumbItemGap = this.thumbnailOptions.gap,
                thumbOuterHeight = thumbItemHeight + thumbItemGap,
                imageLength = this.context.imagesList.length,
                maxAllowedHeight,
                showSlides,
                perPage;

            if (this.isThumbnailsVisible()) {
                maxAllowedHeight = this.galleryWrapper.height();
                perPage = Math.floor(maxAllowedHeight / thumbOuterHeight);
                showSlides = perPage > this.thumbsPerPage ? this.thumbsPerPage : perPage;

                if (showSlides > imageLength) {
                    showSlides = imageLength;
                }

                this.thumbnailsSplide.options.height = showSlides * thumbOuterHeight - thumbItemGap ;
                this.thumbnailsSplide.refresh();
            }

            return this;
        },

        /**
         * Get string with correct background color for the fullscreen
         *
         * @return {string}
         */
        setFsBackground: function () {
            var rgbColor,
                opacity = this.backgroundSettings.opacity || 100,
                color = this.backgroundSettings.color;

            if (this.backgroundSettings.type === 'color') {
                rgbColor = this.hexToRgb(color);

                if (rgbColor) {
                    color = 'rgba(' + rgbColor.full + ',' + opacity / 100 + ')';
                }

                return 'background: ' + color + ';';
            }

            if (this.backgroundSettings.type === 'full_opacity') {
                return 'background: rgba(0,0,0,0);';
            }

            return '';
        },

        /**
         * Convert hex color to the rgb
         *
         * @param {string} hex
         *
         * @return {{r: number, b: number, g: number, array: array, full: string}|null}
         */
        hexToRgb: function (hex) {
            var result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex),
                rChanel = parseInt(result[1], 16),
                gChanel = parseInt(result[2], 16),
                bChanel = parseInt(result[3], 16);

            return result ? {
                r: rChanel,
                g: gChanel,
                b: bChanel,
                array: [rChanel, gChanel, bChanel],
                full: rChanel + ',' + gChanel + ',' + bChanel
            } : null;
        },

        /**
         * Adaptive background
         *
         * @private
         *
         * @param {object} slider
         *
         * @return {void}
         */
        _setAdaptiveBackground: function (slider) {
            this.galleryWrapper.on('gallery:fs:opened', function () {
                this.getAdaptiveColor(slider, this.mainSplide.index);
            }.bind(this));

            slider.on('moved', function (i) {
                this.getAdaptiveColor(slider, i);
            }.bind(this));
        },

        /**
         * Get correct bg color depending on visible image
         * @param {Object} slider
         * @param {number} i
         *
         * @return {void}
         */
        getAdaptiveColor: function (slider, i) {
            var self = this,
                image,
                color,
                style,
                opacity = this.backgroundSettings.opacity || 100,
                fastAverageColor = new FastAverageColor();

            try {
                style = '';
                image = $(slider.Components.Elements.slides[i]).find('img');

                fastAverageColor.getColorAsync(image[0])
                    .then(function (colorData) {
                        if (!$(image).attr('bgStyle')) {
                            color = self.hexToRgb(colorData.hex).full;
                            style += 'rgba(' + color + ',' + opacity / 100 + ')';
                            image.attr('bgStyle', style);
                        }

                        if (self.fsSlidersContainer) {
                            self.fsSlidersContainer.css('background', image.attr('bgStyle'));
                        }
                    })
                    .catch(function (err) {
                        console.log(err);
                    });
            } catch (err) {
                console.log(err);
            }
        }
    };
});
