<?php

declare(strict_types=1);

namespace Amasty\Gallery\ViewModel;

use Amasty\Gallery\Model\Config\Source\FullScreenBackgroundType;
use Amasty\Gallery\Model\Config\Source\FullScreenMainImageChange;
use Amasty\Gallery\Model\Config\Source\MainImageChange;
use Amasty\Gallery\Model\Config\Source\ThumbnailPosition;
use Amasty\Gallery\Model\ConfigProvider;
use Magento\Catalog\Block\Product\View\Gallery;
use Magento\Catalog\Helper\Image;
use Magento\Framework\Config\View;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\View\ConfigInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;

class GalleryViewModel implements ArgumentInterface
{
    const USE_OPTIMIZED_IMAGES = true;
    const DESKTOP_BREAKPOINT = 1024;
    const FULLSCREEN_BREAKPOINT = 768;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ConfigInterface
     */
    private $viewConfig;

    /**
     * @var View|null
     */
    private $configView = null;

    /**
     * @var Gallery
     */
    private $gallery;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Image
     */
    private $imageHelper;

    public function __construct(
        ConfigProvider $configProvider,
        ConfigInterface $viewConfig,
        Gallery $gallery,
        SerializerInterface $serializer,
        Image $imageHelper
    ) {
        $this->configProvider = $configProvider;
        $this->viewConfig = $viewConfig;
        $this->gallery = $gallery;
        $this->serializer = $serializer;
        $this->imageHelper = $imageHelper;
    }

    /**
     * @return string|null
     */
    public function getThumbnailPosition(): ?string
    {
        return $this->configProvider->getThumbnailPosition();
    }

    /**
     * @return bool
     */
    public function isThumbnailsEnabled(): bool
    {
        return $this->configProvider->isShowImageGalleryThumbnails();
    }

    /**
     * @return int
     */
    public function getThumbnailMargin(): int
    {
        return (int)$this->getConfigView()->getVarValue('Magento_Catalog', 'gallery/thumbmargin');
    }

    /**
     * @return int
     */
    public function getThumbnailsWidth(): int
    {
        return (int)$this->gallery->getImageAttribute('product_page_image_small', 'width');
    }

    /**
     * @return int
     */
    public function getThumbnailsHeight(): int
    {
        return (int)$this->gallery->getImageAttribute('product_page_image_small', 'height');
    }

    /**
     * @return string
     */
    public function getBreakpointsConfig(): string
    {
        $amGalleryConfig = [
            'desktopBreakpoint' => self::DESKTOP_BREAKPOINT,
            'fullscreenBreakpoint' =>  self::FULLSCREEN_BREAKPOINT
        ];

        return $this->serializer->serialize($amGalleryConfig);
    }

    /**
     * @return string
     */
    public function getAmGalleryConfig(): string
    {
        $amGalleryConfig = [
            'slidersGap' => $this->getThumbnailMargin(),
            'useOptimizedImages' => self::USE_OPTIMIZED_IMAGES,
            'lazyLoadEnabled' => $this->configProvider->isLazyLoadEnabled(),
            'thumbnailsPosition' => $this->configProvider->getThumbnailPosition(),
            'thumbnailsEnabled' => $this->configProvider->isShowImageGalleryThumbnails(),
            'isFullscreenModeEnabled' => $this->configProvider->isFullscreenModeEnabled(),
            'flipMainImageOnHover' => $this->configProvider->getMainImageChange()
                === MainImageChange::ON_MOUSE_HOVER
        ];

        return (string)$this->serializer->serialize($amGalleryConfig);
    }

    /**
     * @return string
     */
    public function getAmFullscreenConfig(): string
    {
        $amGalleryConfig = [
            'animationEnabled' => $this->configProvider->isAnimationEnabled(),
            'thumbnailsEnabled' => $this->configProvider->isFullscreenShowThumbnails(),
            'thumbnailsPosition' => $this->configProvider->getFullscreenThumbnailsPosition(),
            'flipMainImageOnHover' => $this->configProvider->getFullscreenMainImageChange()
                === FullScreenMainImageChange::ON_HOVER,
            'swipeOnWheelScroll' => $this->configProvider->isMouseScrollActionEnabled(),
            'backgroundType' => $this->configProvider->getFullScreenBackgroundType(),
            'transparentBg' => $this->configProvider->getFullScreenBackgroundType()
                === FullScreenBackgroundType::FULL_OPACITY,
            'adaptiveBg' => $this->configProvider->getFullScreenBackgroundType()
                === FullScreenBackgroundType::ADAPTIVE_BACKGROUND,
            'backgroundColor' => $this->configProvider->getFullscreenBackgroundColor(),
            'backgroundBlur' => $this->configProvider->isFullscreenBackgroundBlur(),
            'backgroundOpacity' => $this->configProvider->getFullScreenOpacityPercentage(),
            'lazyLoadEnabled' => $this->configProvider->isLazyLoadEnabled(),
        ];

        return (string)$this->serializer->serialize($amGalleryConfig);
    }

    /**
     * @return string
     */
    public function getAmZoomConfig(): string
    {
        $amZoomConfig = [
            'zoomActivation' => $this->configProvider->getActivateZoom(),
            'isZoomEnabled' => $this->configProvider->isZoomEnabled(),
            'isFsZoomEnabled' => $this->configProvider->isZoomEnabled()
        ];

        return (string)$this->serializer->serialize($amZoomConfig);
    }

    /**
     * @return string
     */
    public function getImageGalleryType(): string
    {
        return $this->configProvider->getImageGalleryType();
    }

    /**
     * @return View
     */
    private function getConfigView(): View
    {
        if ($this->configView === null) {
            $this->configView = $this->viewConfig->getViewConfig();
        }

        return $this->configView;
    }

    /**
     * @return string
     */
    public function getStyles(): string
    {
        $style = '';
        if ($this->isThumbnailsEnabled() && count($this->gallery->getGalleryImages()->getItems()) > 1) {
            $thumbnailsWidth = $this->getThumbnailsWidth();
            $thumbnailsHeight = $this->getThumbnailsHeight();

            $thumbsPosition = $this->getThumbnailPosition();
            $slidersGap = $this->getThumbnailMargin();

            $previewPaddingVertical = $slidersGap + $thumbnailsWidth;
            $previewPaddingHorizontal = $slidersGap + $thumbnailsHeight;

            if ($thumbsPosition === ThumbnailPosition::LEFT ||
                $thumbsPosition === ThumbnailPosition::RIGHT) {
                $style = 'max-width: calc(100% - ' . $previewPaddingVertical .'px);';
            }

            if ($thumbsPosition === ThumbnailPosition::BOTTOM) {
                $style = 'padding-bottom: ' . $previewPaddingHorizontal .'px;';
            }
        }

        return $style;
    }

    /**
     * @return string
     */
    public function getMainImageData(): string
    {
        $gallery = $this->gallery;
        $images = $gallery->getGalleryImages()->getItems();
        $mainImage = current(array_filter($images, function ($img) use ($gallery) {
            return $gallery->isMainImage($img);
        }));

        if (!empty($images) && empty($mainImage)) {
            $mainImage = $gallery->getGalleryImages()->getFirstItem();
        }

        return $mainImage ?
            (string)$mainImage->getData('medium_image_url') :
            $this->imageHelper->getDefaultPlaceholderUrl('image');
    }
}
