<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Lite Layered Navigation Base for Magento 2
 */

namespace Amasty\ShopbyLite\Helper;

use Magento\Catalog\Api\Data\ProductAttributeInterface;
use Magento\Catalog\Model\Layer\Filter\FilterInterface;
use Magento\Eav\Model\Entity\Attribute\Source\Boolean;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Exception\LocalizedException;

class UrlBuilder extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var FilterInterface
     */
    private $filter;

    /**
     * @var \Amasty\ShopbyLite\Helper\Data
     */
    private $helper;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    public function __construct(
        Context $context,
        \Amasty\ShopbyLite\Helper\Data $helper,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    ) {
        parent::__construct($context);
        $this->helper = $helper;
        $this->storeManager = $storeManager;
    }

    /**
     * @param FilterInterface $filter
     * @param string|array $optionValue
     * @return string
     */
    public function buildUrl(FilterInterface $filter, $optionValue)
    {
        $this->setFilter($filter);

        if ($this->isBooleanFilter($filter) && $optionValue === '0') {
            $query = [$filter->getRequestVar() => null];
        } else {
            $query = [$filter->getRequestVar() => $this->calculateResultValue($optionValue)];
        }

        $query['p'] = null;
        $query['shopbyAjax'] = null;
        $query['_'] = null;

        $params = ['_current' => true, '_use_rewrite' => true, '_query' => $query];
        //fix urls like catalogsearch/result/index/price/10-20/?price=10-60&q=bag
        $params['price'] = null;

        return $this->getUrl('*/*/*', $params);
    }

    /**
     * @param FilterInterface $filter
     * @return bool
     */
    private function isBooleanFilter(FilterInterface $filter): bool
    {
        try {
            $isBoolean = $filter->getAttributeModel()
                && $filter->getAttributeModel()->getSourceModel() === Boolean::class;
        } catch (LocalizedException $e) {
            $isBoolean = false;
        }

        return $isBoolean;
    }

    /**
     * @param FilterInterface $filter
     * @return $this
     */
    public function setFilter(FilterInterface $filter)
    {
        $this->filter = $filter;
        return $this;
    }

    /**
     * @return FilterInterface
     */
    public function getFilter()
    {
        return $this->filter;
    }

    /**
     * @param $route
     * @param array $params
     * @return string
     */
    public function getUrl($route, $params = [])
    {
        return parent::_getUrl($route, $params);
    }

    /**
     * @param FilterInterface $filter
     * @param $resultValue
     * @return array|mixed
     */
    public function buildQuery(FilterInterface $filter, $resultValue)
    {
        return [$filter->getRequestVar() => $resultValue];
    }

    /**
     * @return array
     */
    private function getCurrentValues()
    {
        $values = [];
        $filterCode = $this->filter->getRequestVar();
        if ($this->filter->getRequestVar() == ProductAttributeInterface::CODE_PRICE && $this->useBasePrice()) {
            $filterCode = \Amasty\ShopbyLite\Model\Layer\Filter\Price::AM_BASE_PRICE;
        }

        $data = $this->_request->getParam($filterCode);

        if (!empty($data) || $data === '0') {
            $values = is_array($data) ? $data : explode(',', $data);
            foreach ($values as $key => $value) {
                if (empty($value) && $data !== '0') {
                    unset($values[$key]);
                }
            }
        }

        return $values;
    }

    /**
     * @return bool
     */
    protected function useBasePrice()
    {
        $rate = $this->storeManager->getStore()->getCurrentCurrencyRate();

        return $rate && $rate != 1;
    }

    /**
     * @param $filter
     * @param $optionValue
     * @return string|null
     */
    private function calculateResultValue($optionValue)
    {
        if ($optionValue === null || is_array($optionValue)) {
            return null;
        }

        $key = array_search($optionValue, $this->getCurrentValues());

        if ($this->isMultiSelectAllowed()) {
            $result = $this->getCurrentValues();
            if ($key !== false) {
                unset($result[$key]);
            } else {
                $result[] = $optionValue;
            }
        } else {
            $result = $key !== false ? [] : [$optionValue];
        }

        $value = $result ? implode(',', $result) : null;

        return $value;
    }

    /**
     * @return bool
     */
    private function isMultiSelectAllowed()
    {
        return $this->helper->isMultiSelectAllowed($this->getFilter()->getRequestVar());
    }
}
