<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Amasty\AdvancedMSI\Ui\DataProvider;

use Amasty\AdvancedMSI\Model\GetSourcesBySourceSelectionResult;
use Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;
use Amasty\AdvancedMSI\Model\SourceSelection\CustomerAddressRegister;
use Magento\Framework\Api\Filter;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventoryConfigurationApi\Api\GetStockItemConfigurationInterface;
use Magento\InventorySalesApi\Model\StockByWebsiteIdResolverInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\ItemRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\GetDefaultSourceSelectionAlgorithmCodeInterface;
use Magento\InventorySourceSelectionApi\Api\SourceSelectionServiceInterface;
use Magento\Sales\Model\Order\Item;
use Magento\Sales\Model\OrderRepository;
use Magento\Ui\DataProvider\AbstractDataProvider;
use Magento\InventorySalesApi\Model\GetSkuFromOrderItemInterface;

/**
 * Override
 * @see \Magento\InventoryShippingAdminUi\Ui\DataProvider\SourceSelectionDataProvider
 */
class SourceSelectionDataProvider extends AbstractDataProvider
{
    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var OrderRepository
     */
    private $orderRepository;

    /**
     * @var StockByWebsiteIdResolverInterface
     */
    private $stockByWebsiteIdResolver;

    /**
     * @var GetStockItemConfigurationInterface
     */
    private $getStockItemConfiguration;

    /**
     * @var array
     */
    private $sources = [];

    /**
     * @var SourceSelectionServiceInterface
     */
    private $sourceSelectionService;

    /**
     * @var GetDefaultSourceSelectionAlgorithmCodeInterface
     */
    private $getDefaultSourceSelectionAlgorithmCode;

    /**
     * @var ItemRequestInterfaceFactory
     */
    private $itemRequestFactory;

    /**
     * @var InventoryRequestInterfaceFactory
     */
    private $inventoryRequestFactory;

    /**
     * @var GetSourcesBySourceSelectionResult
     */
    private $getSourcesBySourceSelectionResult;

    /**
     * @var \Magento\InventoryApi\Api\SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var CustomerAddressRegister
     */
    private $addressRegister;

    /**
     * @var GetSkuFromOrderItemInterface
     */
    private $skuFromOrderItem;

    public function __construct(
        $name,
        $primaryFieldName,
        $requestFieldName,
        RequestInterface $request,
        OrderRepository $orderRepository,
        StockByWebsiteIdResolverInterface $stockByWebsiteIdResolver,
        GetStockItemConfigurationInterface $getStockItemConfiguration,
        SourceSelectionServiceInterface $sourceSelectionService,
        GetDefaultSourceSelectionAlgorithmCodeInterface $getDefaultSourceSelectionAlgorithmCode,
        ItemRequestInterfaceFactory $itemRequestFactory,
        InventoryRequestInterfaceFactory $inventoryRequestFactory,
        GetSourcesBySourceSelectionResult $getSourcesBySourceSelectionResult,
        \Magento\InventoryApi\Api\SourceRepositoryInterface $sourceRepository,
        CustomerAddressRegister $addressRegister,
        GetSkuFromOrderItemInterface $skuFromOrderItem,
        array $meta = [],
        array $data = []
    ) {
        parent::__construct($name, $primaryFieldName, $requestFieldName, $meta, $data);
        $this->request = $request;
        $this->orderRepository = $orderRepository;
        $this->stockByWebsiteIdResolver = $stockByWebsiteIdResolver;
        $this->getStockItemConfiguration = $getStockItemConfiguration;
        $this->sourceSelectionService = $sourceSelectionService;
        $this->getDefaultSourceSelectionAlgorithmCode = $getDefaultSourceSelectionAlgorithmCode;
        $this->itemRequestFactory = $itemRequestFactory;
        $this->inventoryRequestFactory = $inventoryRequestFactory;
        $this->getSourcesBySourceSelectionResult = $getSourcesBySourceSelectionResult;
        $this->sourceRepository = $sourceRepository;
        $this->addressRegister = $addressRegister;
        $this->skuFromOrderItem = $skuFromOrderItem;
    }

    /**
     * Disable for collection processing | ????
     *
     * @param Filter $filter
     * @return bool
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function addFilter(Filter $filter)
    {
        return null;
    }

    /**
     * override of
     * @see \Magento\InventoryShippingAdminUi\Ui\DataProvider\SourceSelectionDataProvider::getData
     * {@inheritdoc}
     */
    public function getData()
    {
        $data = [];
        $orderId = $this->request->getParam('order_id');
        /** @var \Magento\Sales\Model\Order $order */
        $order = $this->orderRepository->get($orderId);
        $websiteId = $order->getStore()->getWebsiteId();
        //TODO advanced stock resolver
        $stockId = (int)$this->stockByWebsiteIdResolver->execute((int)$websiteId)->getStockId();

        foreach ($order->getAllItems() as $orderItem) {
            if ($orderItem->getIsVirtual()
                || $orderItem->getLockedDoShip()
                || $orderItem->getHasChildren()) {
                continue;
            }

            $orderItemId = $orderItem->getId();
            if ($orderItem->getParentItem() && !$orderItem->isShipSeparately()) {
                $orderItemId = $orderItem->getParentItemId();
            }

            $qty = $orderItem->getSimpleQtyToShip();
            $qty = $this->castQty($orderItem, $qty);
            $sku = $this->skuFromOrderItem->execute($orderItem);

            $data[$orderId]['items'][] = [
                'orderItemId' => $orderItemId,
                'sku' => $sku,
                'product' => $this->getProductName($orderItem),
                'qtyToShip' => $qty,
                'sources' => [],
                'isManageStock' => $this->isManageStock($sku, $stockId)
            ];
        }
        $data[$orderId]['websiteId'] = $websiteId;
        $data[$orderId]['order_id'] = $orderId;

        $this->assignSources($data[$orderId], $stockId);

        return $data;
    }

    /**
     * @param array $data
     * @param int $stockId
     */
    protected function assignSources(array &$data, int $stockId): void
    {
        $requestItems = [];
        foreach ($data['items'] as $dataRow) {
            $requestItems[] = $this->itemRequestFactory->create(
                [
                    'sku' => $dataRow['sku'],
                    'qty' => $dataRow['qtyToShip']
                ]
            );
        }

        /** @var \Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface $sourceSelectionResult */
        $inventoryRequest = $this->inventoryRequestFactory->create([
            'stockId' => $stockId,
            'items'   => $requestItems
        ]);
        $algorithmCode = $this->getDefaultSourceSelectionAlgorithmCode->execute();

        if ($algorithmCode == 'distance' || $algorithmCode == Combined::CODE) {
            $this->saveDestinationAddress((int)$data['order_id']);
        }
        $sourceSelectionResult = $this->sourceSelectionService->execute(
            $inventoryRequest,
            $algorithmCode
        );

        foreach ($data['items'] as &$dataRow) {
            $dataRowSources = [];

            foreach ($sourceSelectionResult->getSourceSelectionItems() as $item) {
                $sourceCode = $item->getSourceCode();

                if ($item->getSku() === $dataRow['sku']
                    && !isset($dataRowSources[$sourceCode])
                ) {
                    $dataRowSources[$sourceCode] = [
                        'sourceName' => $this->getSourceName($sourceCode),
                        'sourceCode' => $sourceCode,
                        'qtyAvailable' => $item->getQtyAvailable(),
                        'qtyToDeduct' => $item->getQtyToDeduct()
                    ];
                }
            }

            $dataRow['sources'] = array_values($dataRowSources);
        }

        $sourceCodes = $this->getSourcesBySourceSelectionResult->execute($sourceSelectionResult);
        foreach ($sourceCodes as $sourceCode) {
            $data['sourceCodes'][] = [
                'value' => $sourceCode,
                'label' => $this->getSourceName((string)$sourceCode)
            ];
        }
    }

    /**
     * Get source name by code
     *
     * @param string $sourceCode
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getSourceName(string $sourceCode): string
    {
        if (!isset($this->sources[$sourceCode])) {
            $this->sources[$sourceCode] = $this->sourceRepository->get($sourceCode)->getName();
        }

        return $this->sources[$sourceCode];
    }

    /**
     * @param string $itemSku
     * @param int $stockId
     * @return bool
     * @throws LocalizedException
     */
    private function isManageStock($itemSku, $stockId)
    {
        $stockItemConfiguration = $this->getStockItemConfiguration->execute($itemSku, $stockId);

        return $stockItemConfiguration->isManageStock();
    }

    /**
     * Generate display product name
     * @param Item $item
     * @return null|string
     */
    private function getProductName(Item $item)
    {
        //TODO: need to transfer this to html block and render on Ui
        $name = $item->getName();
        if ($parentItem = $item->getParentItem()) {
            $name = $parentItem->getName();
            $options = [];
            if ($productOptions = $parentItem->getProductOptions()) {
                if (isset($productOptions['options'])) {
                    $options = array_merge($options, $productOptions['options']);
                }
                if (isset($productOptions['additional_options'])) {
                    $options = array_merge($options, $productOptions['additional_options']);
                }
                if (isset($productOptions['attributes_info'])) {
                    $options = array_merge($options, $productOptions['attributes_info']);
                }
                if (count($options)) {
                    foreach ($options as $option) {
                        $name .= '<dd>' . $option['label'] . ': ' . $option['value'] .'</dd>';
                    }
                } else {
                    $name .= '<dd>' . $item->getName() . '</dd>';
                }
            }
        }

        return $name;
    }

    /**
     * @param Item $item
     * @param string|int|float $qty
     * @return float|int
     */
    private function castQty(Item $item, $qty)
    {
        if ($item->getIsQtyDecimal()) {
            $qty = (double)$qty;
        } else {
            $qty = (int)$qty;
        }

        return $qty > 0 ? $qty : 0;
    }

    /**
     * @param int $orderId
     *
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function saveDestinationAddress($orderId)
    {
        /** @var \Magento\Sales\Model\Order\Address $shippingAddress */
        $shippingAddress = $this->orderRepository->get($orderId)->getShippingAddress();

        $this->addressRegister->setDestinationAddress(
            (string)$shippingAddress->getCountryId(),
            (string)$shippingAddress->getPostcode(),
            (string)$shippingAddress->getStreet()[0],
            (string)$shippingAddress->getRegion(),
            (string)$shippingAddress->getCity()
        );
    }
}
