<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Ui\DataProvider\Listing;

use Magento\Framework\App\RequestInterface;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryLowQuantityNotificationApi\Api\Data\SourceItemConfigurationInterface;
use Magento\Inventory\Model\ResourceModel\Source\CollectionFactory;
use Magento\Inventory\Model\ResourceModel\Source\Collection;
use Magento\Inventory\Model\SourceItem\Command\GetSourceItemsBySku;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceItemResource;

class SourceDataProvider extends \Magento\Ui\DataProvider\AbstractDataProvider
{
    /**
     * @var string
     */
    private $productSku;

    /**
     * @var string
     */
    private $productId;

    /**
     * @var string
     */
    private $productTypeId;

    /**
     * @var GetSourceItemsBySku
     */
    private $getSourceItemsBySku;

    /**
     * @var Collection
     */
    protected $collection;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var SourceItemResource
     */
    private $sourceItemResource;

    public function __construct(
        $name,
        $primaryFieldName,
        $requestFieldName,
        CollectionFactory $collectionFactory,
        GetSourceItemsBySku $getSourceItemsBySku,
        SourceItemResource $sourceItemResource,
        RequestInterface $request,
        array $meta = [],
        array $data = []
    ) {
        parent::__construct($name, $primaryFieldName, $requestFieldName, $meta, $data);
        $this->collectionFactory = $collectionFactory;
        $this->collection = $collectionFactory->create();
        $this->getSourceItemsBySku = $getSourceItemsBySku;
        $this->productSku = $request->getParam('sku');
        $this->productId = $request->getParam('entity_id');
        $this->productTypeId = $request->getParam('type_id');
        $this->sourceItemResource = $sourceItemResource;
    }

    /**
     * Get Data
     *
     * @return array
     *
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getData()
    {
        if (!$this->productSku) {
            return parent::getData();
        }

        $data = parent::getData();

        /** @var \Magento\InventoryApi\Api\Data\SourceItemInterface $item */
        foreach ($data['items'] as &$item) {
            if (!isset($item['source_code'])) {
                continue;
            }
            $item['sku'] = $this->productSku;
            $item['product_id'] = $this->productId;
            $item['type_id'] = $this->productTypeId;

            $itemData = $this->sourceItemResource->getSourceItemData($this->productSku, $item['source_code']);
            $item['quantity'] = isset($itemData['quantity']) ? (float)$itemData['quantity'] : 0;
            $item['source_item_status'] =
                !isset($itemData['status']) ? SourceItemInterface::STATUS_IN_STOCK : $itemData['status'];

            $notifyData = $this->sourceItemResource->getNotifyQtyForProduct($this->productSku, $item['source_code']);
            if (isset($notifyData['use_default'])) {
                $item['notify_stock_qty'] = (float)$notifyData['use_default'];
                $item['initial_notify_stock_qty'] = null;
            } else {
                $item['notify_stock_qty'] = (float)$notifyData['notify_stock_qty'];
            }
        }

        $data['totalRecords'] += 1; // temp solution for magento issue https://github.com/magento/magento2/issues/25457

        return $data;
    }
}
