<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Plugin\Shipping\Model;

use Amasty\AdvancedMSI\Api\SourceCustomShippingRateRepositoryInterface;
use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\SourceItem\GetSourceItemsSortedByStock;
use Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;
use Amasty\AdvancedMSI\Model\SourceSelection\CustomerAddressRegister;
use Amasty\AdvancedMSI\Model\SourceSelection\SourceInfoRegister;
use Amasty\Base\Model\Di\Wrapper as IsPreorder;
use Amasty\Base\Model\Di\Wrapper as PreorderConfigProvider;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Model\Product\Configuration\Item\ItemInterface;
use Magento\CatalogInventory\Model\Stock;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use Magento\InventoryConfiguration\Model\GetStockItemConfiguration;
use Magento\InventorySalesApi\Model\StockByWebsiteIdResolverInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\ItemRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\SourceSelectionServiceInterface;
use Magento\Quote\Api\Data\CartItemInterface;
use Magento\Quote\Model\Quote\Address\Item as QuoteAddressItem;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Shipping\Model\Shipping;
use Magento\Store\Model\ScopeInterface;

/**
 * @TODO Decompose this plugin. Transferring business logic to models
 */
class ShippingPlugin
{
    public const PACKAGE_WEIGHT = 'package_weight';
    public const PACKAGE_QTY = 'package_qty';
    public const SKU = 'sku';
    public const QTY = 'qty';
    public const AM_SHIPPING_TABLE_RATE = 'amstrates';

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var InventoryRequestInterfaceFactory
     */
    private $inventoryRequestInterfaceFactory;

    /**
     * @var SourceSelectionServiceInterface
     */
    private $sourceSelectionService;

    /**
     * @var ItemRequestInterfaceFactory
     */
    private $itemRequestFactory;

    /**
     * @var StockByWebsiteIdResolverInterface
     */
    private $stockByWebsiteIdResolver;

    /**
     * @var SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var CustomerAddressRegister
     */
    private $addressRegister;

    /**
     * @var SourceInfoRegister
     */
    private $sourceInfoRegister;

    /**
     * @var SourceCustomShippingRateRepositoryInterface
     */
    private $sourceCustomShippingRateRepository;

    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;

    /**
     * @var GetSourcesAssignedToStockOrderedByPriorityInterface
     */
    private $sourcesAssignedToStockOrderedByPriority;

    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;

    /**
     * @var GetStockItemConfiguration
     */
    private $getStockItemConfiguration;

    /**
     * @var GetSourceItemsSortedByStock
     */
    private $sourceItemsSortedByStock;

    /**
     * @var PreorderConfigProvider
     */
    private $preorderConfigProvider;

    /**
     * @var IsPreorder
     */
    private $isPreorder;

    public function __construct(
        ConfigProvider $configProvider,
        InventoryRequestInterfaceFactory $inventoryRequestInterfaceFactory,
        SourceSelectionServiceInterface $sourceSelectionService,
        ItemRequestInterfaceFactory $itemRequestFactory,
        StockByWebsiteIdResolverInterface $stockByWebsiteIdResolver,
        SourceRepositoryInterface $sourceRepository,
        ScopeConfigInterface $scopeConfig,
        CustomerAddressRegister $addressRegister,
        SourceInfoRegister $sourceInfoRegister,
        SourceCustomShippingRateRepositoryInterface $sourceCustomShippingRateRepository,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        GetSourcesAssignedToStockOrderedByPriorityInterface $sourcesAssignedToStockOrderedByPriority,
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        GetStockItemConfiguration $getStockItemConfiguration,
        GetSourceItemsSortedByStock $sourceItemsSortedByStock,
        PreorderConfigProvider $preorderConfigProvider,
        IsPreorder $isPreorder
    ) {
        $this->configProvider = $configProvider;
        $this->inventoryRequestInterfaceFactory = $inventoryRequestInterfaceFactory;
        $this->sourceSelectionService = $sourceSelectionService;
        $this->itemRequestFactory = $itemRequestFactory;
        $this->stockByWebsiteIdResolver = $stockByWebsiteIdResolver;
        $this->sourceRepository = $sourceRepository;
        $this->scopeConfig = $scopeConfig;
        $this->addressRegister = $addressRegister;
        $this->sourceInfoRegister = $sourceInfoRegister;
        $this->sourceCustomShippingRateRepository = $sourceCustomShippingRateRepository;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->sourcesAssignedToStockOrderedByPriority = $sourcesAssignedToStockOrderedByPriority;
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->getStockItemConfiguration = $getStockItemConfiguration;
        $this->sourceItemsSortedByStock = $sourceItemsSortedByStock;
        $this->preorderConfigProvider = $preorderConfigProvider;
        $this->isPreorder = $isPreorder;
    }

    public function aroundCollectRates(
        Shipping $subject,
        \Closure $proceed,
        RateRequest $request
    ): Shipping {
        $quoteItems = $request->getAllItems();

        if (!$this->configProvider->isEnabled()
            || !$this->configProvider->getShippingCostOnSourceEnabled()
            || empty($quoteItems)
        ) {
            return $proceed($request);
        }

        $requestItems = [];
        $requestItemsInfo = [];
        $algorithm = $this->configProvider->getDefaultAlgorithm();
        $stockId = (int)$this->stockByWebsiteIdResolver->execute((int)$request->getWebsiteId())->getStockId();
        $configurableQty = '';
        $configurableId = '';

        foreach ($quoteItems as $item) {
            if ($item->getProductType() === Configurable::TYPE_CODE) {
                $configurableQty = $item->getQty();
                $configurableId = $item->getItemId();
                continue;
            }
            $qty = $item->getParentItemId() === $configurableId ? $configurableQty : $item->getQty();

            $productSku = $this->getSku($item);

            $itemQty = $item->getQty();
            $itemWeight = $itemQty *  $item->getWeight();

            if (!isset($requestItemsInfo[$productSku])) {
                $requestItemsInfo[$productSku] = [self::PACKAGE_WEIGHT => $itemWeight, self::PACKAGE_QTY => $itemQty];
            } else {
                $requestItemsInfo[$productSku][self::PACKAGE_WEIGHT] += $itemWeight;
                $requestItemsInfo[$productSku][self::PACKAGE_QTY] += $itemQty;
            }

            if (!isset($requestItems[$productSku])) {
                $requestItems[$productSku] = $this->itemRequestFactory->create(
                    [
                        self::SKU => $productSku,
                        self::QTY => $qty
                    ]
                );
            } else {
                $requestedQty = $requestItems[$productSku]->getQty();
                $requestItems[$productSku]->setQty($requestedQty + $qty);
            }
        }

        /** @var InventoryRequestInterface $sourceSelectionResult */
        $inventoryRequest = $this->inventoryRequestInterfaceFactory->create([
            'stockId' => $stockId,
            'items'   => $requestItems
        ]);

        $sourceSelectionResult = $this->getSourceSelectionResult($quoteItems, $inventoryRequest, $algorithm);
        $this->sourceInfoRegister->setSourceSelectionResult($sourceSelectionResult);

        $sources = [];
        $itemsBySource = [];
        $sourceSelectionItems = $sourceSelectionResult->getSourceSelectionItems();
        foreach ($sourceSelectionItems as $item) {
            $itemsBySource[$item->getSourceCode()][] = $item->getSku();
        }
        foreach ($sourceSelectionItems as $item) {
            if ($item->getQtyToDeduct() != 0 && !array_key_exists($item->getSourceCode(), $sources)) {
                /** @var SourceInterface $source */
                $source = $this->sourceRepository->get($item->getSourceCode());
                $subject->resetResult();
                $this->setRequestData($request, $source);
                if (isset($requestItemsInfo[$item->getSku()])) {
                    $this->setRequestItemData($request, $requestItemsInfo[$item->getSku()]);
                }
                $this->getCarrierRates($subject, $request, $itemsBySource[$item->getSourceCode()]);
                $allRates = $subject->getResult()->getAllRates();
                $sources[$item->getSourceCode()] = $allRates;
            }
        }

        return $this->combineDuplicatesAndSetCustomShippingRate($subject, $sources);
    }

    /**
     * Compatibility with negative backorder threshold.
     * Compatibility with Amasty_Preorder. Add stock items of all pre ordered products.
     */
    private function getSourceSelectionResult(
        array $quoteItems,
        InventoryRequestInterface $inventoryRequest,
        string $algorithm
    ): SourceSelectionResultInterface {
        $sourceSelectionResult = $this->sourceSelectionService->execute(
            $inventoryRequest,
            $algorithm
        );

        $sourceSelectionItems = $sourceSelectionResult->getSourceSelectionItems();
        $preparedSourceSelectionItems = [];
        $itemsTdDeliver = [];

        foreach ($quoteItems as $item) {
            $itemsTdDeliver[$item->getSku()] = $item->getQty();

            foreach ($sourceSelectionItems as $sourceSelectionItem) {
                if ($sourceSelectionItem->getSku() === $this->getSku($item)) {
                    $stockItem = $this->getStockItemConfiguration->execute(
                        $item->getSku(),
                        $inventoryRequest->getStockId()
                    );

                    if ($stockItem->isManageStock()
                        && in_array(
                            $stockItem->getBackorders(),
                            [Stock::BACKORDERS_YES_NONOTIFY, Stock::BACKORDERS_YES_NOTIFY]
                        )
                    ) {
                        $sourceQtyAvailable = $sourceSelectionItem->getQtyAvailable() - $stockItem->getMinQty();
                        $qtyToDeduct = min($sourceQtyAvailable, $itemsTdDeliver[$item->getSku()] ?? 0.0);

                        $preparedSourceSelectionItems[] = $this->sourceSelectionItemFactory->create([
                            'sourceCode' => $sourceSelectionItem->getSourceCode(),
                            'sku' => $item->getSku(),
                            'qtyToDeduct' => $qtyToDeduct,
                            'qtyAvailable' => $sourceQtyAvailable
                        ]);
                    } else {
                        $preparedSourceSelectionItems[] = $sourceSelectionItem;
                    }

                    $itemsTdDeliver[$item->getSku()] -= $qtyToDeduct ?? $sourceSelectionItem->getQtyToDeduct();
                }
            }
        }

        $isShippable = true;

        /**
         * Compare the number of items to be delivered with some epsilon
         * @see \Magento\InventorySourceSelectionApi\Model\Algorithms\Result\GetDefaultSortedSourcesResult::isZero
         */
        foreach ($itemsTdDeliver as $itemToDeliver) {
            if ($itemToDeliver >= 0.0000001) {
                $isShippable = false;
                break;
            }
        }

        if ($this->preorderConfigProvider->isEnabled()) {
            $preorderSourceItems = [];
            $sourceCodes = $this->getEnabledSourcesOrderedByPriorityByStockId(
                $inventoryRequest->getStockId()
            );

            foreach ($quoteItems as $item) {
                if ($item instanceof QuoteAddressItem && $item->getQuoteItem()) {
                    $item = $item->getQuoteItem();
                }

                if ($this->isPreorder->execute($item)) {
                    $sourceItem = $this->sourceItemsSortedByStock
                        ->getSourceItemBySourceCodeAndSku($sourceCodes, $item->getSku());

                    if ($sourceItem) {
                        $preorderSourceItems[] = $this->sourceSelectionItemFactory->create([
                            'sourceCode' => $sourceItem->getSourceCode(),
                            'sku' => $item->getSku(),
                            'qtyToDeduct' => $item->getQty(),
                            'qtyAvailable' => 0
                        ]);
                    }
                }
            }

            if ($preorderSourceItems) {
                $preparedSourceSelectionItems = array_merge(
                    $preparedSourceSelectionItems,
                    $preorderSourceItems
                );
                $isShippable = true;
            }
        }

        if ($preparedSourceSelectionItems) {
            $sourceSelectionResult = $this->sourceSelectionResultFactory->create([
                'sourceItemSelections' => $preparedSourceSelectionItems,
                'isShippable' => $isShippable
            ]);
        }

        return $sourceSelectionResult;
    }

    private function getEnabledSourcesOrderedByPriorityByStockId(int $stockId): array
    {
        $sources = $this->sourcesAssignedToStockOrderedByPriority->execute($stockId);
        $sources = array_filter(
            $sources,
            function (SourceInterface $source) {
                return $source->isEnabled();
            }
        );
        $sourceCodes = [];

        foreach ($sources as $source) {
            $sourceCodes[] = $source->getSourceCode();
        }

        return $sourceCodes;
    }

    public function setRequestData(RateRequest $request, SourceInterface $source): void
    {
        if (!$request->getOrig()) {
            $request->setCountryId(
                $source->getCountryId()
            )->setRegionId(
                $source->getRegion()
            )->setCity(
                $source->getCity()
            )->setPostcode(
                $source->getPostcode()
            )->setOrigPostcode(
                $source->getPostcode()
            )->setOrigRegionCode(
                $source->getRegionId()
            )->setOrigCountry(
                $source->getCountryId()
            );
        }
    }

    public function setRequestItemData(RateRequest $request, array $requestItem): void
    {
        if (!$request->getOrig()) {
            $request->setPackageWeight(
                $requestItem[self::PACKAGE_WEIGHT]
            )->setPackageQty(
                $requestItem[self::PACKAGE_QTY]
            );
        }
    }

    public function getCarrierRates(Shipping $subject, RateRequest $request, array $itemsBySource = []): void
    {
        $storeId = $request->getStoreId();
        $limitCarrier = $request->getLimitCarrier();

        if (!$limitCarrier) {
            $carriers = $this->scopeConfig->getValue(
                'carriers',
                ScopeInterface::SCOPE_STORE,
                $storeId
            );

            foreach ($carriers as $carrierCode => $carrierConfig) {
                // To calculate rates for products from one source we should use all of them at the same time.
                // Compatibility with Amasty_ShippingTableRates.
                if ($carrierCode === self::AM_SHIPPING_TABLE_RATE) {
                    $newItems = [];
                    $oldItems = $request->getAllItems();
                    foreach ($oldItems as $oldItem) {
                        if (in_array($oldItem->getSku(), $itemsBySource)) {
                            $newItems[] = $oldItem;
                        }
                    }
                    $request->setAllItems($newItems);
                    $subject->collectCarrierRates($carrierCode, $request);
                    $request->setAllItems($oldItems);
                } else {
                    $subject->collectCarrierRates($carrierCode, $request);
                }
            }
        } else {
            if (!is_array($limitCarrier)) {
                $limitCarrier = [$limitCarrier];
            }
            foreach ($limitCarrier as $carrierCode) {
                $carrierConfig = $this->scopeConfig->getValue(
                    'carriers/' . $carrierCode,
                    ScopeInterface::SCOPE_STORE,
                    $storeId
                );
                if (!$carrierConfig) {
                    continue;
                }
                $subject->collectCarrierRates($carrierCode, $request);
            }
        }
    }

    public function combineDuplicatesAndSetCustomShippingRate(Shipping $subject, array $sources): Shipping
    {
        $ratesPrices = [];
        $allRates = [];

        $this->filterAmrates($sources);
        foreach ($sources as $sourceCode => $rates) {
            $customRates = $this->sourceCustomShippingRateRepository->getRatesBySourceCode($sourceCode);
            foreach ($rates as $rate) {
                $methodName = $rate->getData('method');
                $price = isset($customRates[$methodName])
                    ? $customRates[$methodName]->getShippingRate()
                    : $rate->getData('price');
                if (!array_key_exists($methodName, $ratesPrices)) {
                    $ratesPrices[$methodName] = $price;
                    $allRates[] = $rate;
                } else {
                    $ratesPrices[$methodName] += $price;
                }
            }
        }

        $subject->resetResult();

        foreach ($allRates as $rate) {
            $rate->setPrice($ratesPrices[$rate->getData('method')]);
            $subject->getResult()->append($rate);
        }

        return $subject;
    }

    /**
     * @param CartItemInterface|ItemInterface $item
     * @return string
     */
    private function getSku($item): string
    {
        return $item->getProduct()->getData(ProductInterface::SKU) ?: $item->getSku();
    }

    /**
     * Unset amrates for source if it is skipped for any product on other source.
     *
     * @param array $sources ['source_code' => [
     * 0 => \Magento\Quote\Model\Quote\Address\RateResult\Method
     * 1 => \Magento\Quote\Model\Quote\Address\RateResult\Method
     * ]]
     */
    private function filterAmrates(array &$sources): void
    {
        $compare = [];
        foreach ($sources as $sourceCode => $rates) {
            foreach ($rates as $key => $rate) {
                $compare[$sourceCode] = [];
                if ($rate->getCarrier() === self::AM_SHIPPING_TABLE_RATE) {
                    $compare[$sourceCode][$rate->getMethod()] = $key;
                }
            }
        }
        if (!empty($compare) && count($compare) > 1) {
            uasort($compare, function ($a, $b) {
                return count($b) - count($a);
            });
            $diff = array_diff_key(...(array_values($compare)));

            foreach ($compare as $sourceCode => $rates) {
                foreach (array_keys($diff) as $key) {
                    if (isset($rates[$key])) {
                        unset($sources[$sourceCode][$rates[$key]]);
                    }
                }
            }
        }
    }
}
