<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Plugin\InventorySourceUi;

use Magento\Ui\Component\Container;
use Magento\Ui\Component\Form\Field;
use Magento\Ui\Component\Form\Element\Input;
use Amasty\AdvancedMSI\Api\SourceCustomShippingRateRepositoryInterface;
use Amasty\AdvancedMSI\Api\Data\SourceEmailInterface;
use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\CollectionFactory;

class SourceDataProvider
{
    public const SHIPMENT_EMAILS = 'shipment_emails';

    public const LOW_STOCK_EMAILS = 'low_stock_emails';

    public const RATE_COMMENT = 'amasty_rate_comment';

    /**
     * @var \Magento\Shipping\Model\Config
     */
    private $shippingConfig;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var \Magento\Framework\App\RequestInterface
     */
    private $request;

    /**
     * @var \Amasty\AdvancedMSI\Api\Data\SourceCustomShippingRateInterface[]
     */
    private $sourceCustomShippingRates;

    /**
     * @var string
     */
    private $sourceCode;

    /**
     * @var SourceCustomShippingRateRepositoryInterface
     */
    private $customShippingRateRepository;

    /**
     * @var CollectionFactory
     */
    private $emailCollectionFactory;
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        \Magento\Shipping\Model\Config $shippingConfig,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Amasty\AdvancedMSI\Api\SourceCustomShippingRateRepositoryInterface $customShippingRateRepository,
        \Magento\Framework\App\RequestInterface $request,
        CollectionFactory $emailCollectionFactory,
        ConfigProvider $configProvider
    ) {
        $this->shippingConfig = $shippingConfig;
        $this->scopeConfig = $scopeConfig;
        $this->request = $request;
        $this->customShippingRateRepository = $customShippingRateRepository;
        $this->emailCollectionFactory = $emailCollectionFactory;
        $this->configProvider = $configProvider;
    }

    public function afterGetData(
        \Magento\InventoryAdminUi\Ui\DataProvider\SourceDataProvider $subject,
        array $result
    ) {
        if (\Magento\InventoryAdminUi\Ui\DataProvider\SourceDataProvider::SOURCE_FORM_NAME === $subject->getName()) {
            if (!empty($this->sourceCustomShippingRates)) {
                foreach ($this->sourceCustomShippingRates as $sourceCustomShippingRate) {
                    $result[$this->sourceCode]['general']['custom_shipping_rate']
                        [$sourceCustomShippingRate->getShippingRateCode()]
                            = $sourceCustomShippingRate->getShippingRate();
                }
            }
            $orderEmails = implode(',', $this->emailCollectionFactory->create()
                ->getEmailsByType($this->sourceCode, SourceDataProvider::SHIPMENT_EMAILS));
            $lowStockEmails = implode(',', $this->emailCollectionFactory->create()
                ->getEmailsByType($this->sourceCode, SourceDataProvider::LOW_STOCK_EMAILS));
            $result[$this->sourceCode]['general'][SourceDataProvider::SHIPMENT_EMAILS] = $orderEmails;
            $result[$this->sourceCode]['general'][SourceDataProvider::LOW_STOCK_EMAILS] = $lowStockEmails;
        }

        return $result;
    }

    public function afterGetMeta(
        \Magento\InventoryAdminUi\Ui\DataProvider\SourceDataProvider $subject,
        array $result
    ) {
        $this->sourceCode = $this->request->getParam('source_code');
        if (\Magento\InventoryAdminUi\Ui\DataProvider\SourceDataProvider::SOURCE_FORM_NAME === $subject->getName()) {
            $this->addCustomShippingRatesTab($result);
            if (!$this->configProvider->getShippingCostOnSourceEnabled()) {
                $this->addComment($result);
            }
            $this->addEmails($result, self::SHIPMENT_EMAILS, __('Shipment Email Notification'));
            $this->addEmails($result, self::LOW_STOCK_EMAILS, __('Low Stock Notification'));
            $this->checkProductsTabListing($result);
        }

        return $result;
    }

    public function addCustomShippingRatesTab(array &$result)
    {
        $this->sourceCustomShippingRates = $this->customShippingRateRepository->getRatesBySourceCode(
            $this->sourceCode
        );
        $customRatesIsEnabled = $this->configProvider->getShippingCostOnSourceEnabled();

        foreach ($this->shippingConfig->getActiveCarriers() as $carrierCode => $carrier) {
            if ($carrierMethods = $carrier->getAllowedMethods()) {
                $result['custom_shipping_rates']['children']
                ['carrier_' . $carrier->getCarrierCode()]['arguments']['data']['config'] = [
                    'label' => $this->scopeConfig->getValue('carriers/' . $carrierCode . '/title'),
                    'collapsible' => true,
                    'opened' => false,
                    'visible' => $customRatesIsEnabled,
                    'componentType' => 'fieldset'
                ];
                foreach ($carrierMethods as $methodCode => $method) {
                    $result['custom_shipping_rates']['children']
                    ['carrier_' . $carrier->getCarrierCode()]['children']['custom_shipping_rate.' . $methodCode]
                    ['arguments']['data']['config'] = [
                        'label' => $method,
                        'dataType' => 'text',
                        'formElement' => 'input',
                        'componentType' => 'field',
                        'service' => ['template' => 'Amasty_AdvancedMSI/form/element/helper/service'],
                        'disabled' => !isset($this->sourceCustomShippingRates[$methodCode]),
                        'source' => 'custom_shipping_rate.' . $methodCode
                    ];
                }
            }
        }
    }

    public function addComment(array &$result)
    {
        $result['custom_shipping_rates']['children'][self::RATE_COMMENT]['arguments']['data']['config'] = [
            'componentType' => Container::NAME,
            'component' => 'Magento_Ui/js/form/components/html',
            'additionalClasses' => 'admin__fieldset-note',
            'content' => __(
                'Please mind that \'Automatic Source Selection\' should be enabled to let 
                Custom Shipping Rates been applied and charged to a customer on a checkout step. Please navigate to 
                Stores > Configuration > Amasty Extensions > Advanced MSI > Automatic Source Selection & Shipping Cost 
                Calculating to check if the setting is YES.'
            ),
            'sortOrder' => 0
        ];
    }

    public function checkProductsTabListing(array &$result)
    {
        if (!$this->sourceCode) {
            $result['product_tab']['arguments']['data']['config']['visible'] = false;
        }
    }

    public function addEmails(array &$result, $key, $title)
    {
        $result = array_replace_recursive(
            $result,
            [
                'general' => [
                    'children' => [
                        $key => [
                            'arguments' => [
                                'data' => [
                                    'config' => [
                                        'label' => $title,
                                        'componentType' => Field::NAME,
                                        'component' => 'Amasty_AdvancedMSI/js/form/element/email',
                                        'formElement' => Input::NAME,
                                        'dataScope' => $key,
                                        'sortOrder' => 70
                                    ],
                                ],
                            ],
                        ],
                    ],
                ]
            ]
        );

        return $result;
    }
}
