<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Plugin\InventoryShippingAdminUi\Controller\Adminhtml\SourceSelection;

use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\Repository\ReservedProductsRepository;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\CollectionFactory;
use Amasty\AdvancedMSI\Utils\Email;
use Magento\Framework\App\Response\RedirectInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface;
use Magento\InventoryApi\Api\GetSourceItemsBySkuInterface;
use Magento\InventoryApi\Api\SourceItemsSaveInterface;
use Magento\InventoryLowQuantityNotification\Model\ResourceModel\LowQuantityCollection;
use Magento\InventorySalesApi\Model\GetSkuFromOrderItemInterface;
use Magento\Sales\Api\Data\ShipmentExtensionFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Convert\Order;
use Magento\Sales\Model\Order as OrderModel;
use Magento\Sales\Model\Order\ShipmentRepository;

class ShipmentPlugin
{
    /**
     * @var ReservedProductsRepository
     */
    private $reservedProductsRepository;

    /**
     * @var GetSourceItemsBySkuInterface
     */
    private $sourceItemsBySku;

    /**
     * @var SourceItemsSaveInterface
     */
    private $sourceItemsSave;

    /**
     * @var ManagerInterface
     */
    private $messageManager;

    /**
     * @var RedirectInterface
     */
    private $redirect;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var Order
     */
    private $convertOrder;

    /**
     * @var ShipmentExtensionFactory
     */
    private $shipmentExtensionFactory;

    /**
     * @var ShipmentRepository
     */
    private $shipmentRepository;
    /**
     * @var CollectionFactory
     */
    private $emailCollectionFactory;
    /**
     * @var Email
     */
    private $emailSender;
    /**
     * @var LowQuantityCollection
     */
    private $lowQuantityCollection;

    /**
     * @var GetSkuFromOrderItemInterface
     */
    private $skuFromOrderItem;

    public function __construct(
        ReservedProductsRepository $reservedProductsRepository,
        ConfigProvider $configProvider,
        GetSourceItemsBySkuInterface $sourceItemsBySku,
        SourceItemsSaveInterface $sourceItemsSave,
        ManagerInterface $messageManager,
        RedirectInterface $redirect,
        OrderRepositoryInterface $orderRepository,
        Order $convertOrder,
        ShipmentExtensionFactory $shipmentExtensionFactory,
        ShipmentRepository $shipmentRepository,
        CollectionFactory $emailCollectionFactory,
        Email $emailSender,
        LowQuantityCollection $lowQuantityCollection,
        GetSkuFromOrderItemInterface $skuFromOrderItem
    ) {
        $this->reservedProductsRepository = $reservedProductsRepository;
        $this->sourceItemsBySku = $sourceItemsBySku;
        $this->sourceItemsSave = $sourceItemsSave;
        $this->messageManager = $messageManager;
        $this->redirect = $redirect;
        $this->configProvider = $configProvider;
        $this->orderRepository = $orderRepository;
        $this->convertOrder = $convertOrder;
        $this->shipmentExtensionFactory = $shipmentExtensionFactory;
        $this->shipmentRepository = $shipmentRepository;
        $this->emailCollectionFactory = $emailCollectionFactory;
        $this->emailSender = $emailSender;
        $this->lowQuantityCollection = $lowQuantityCollection;
        $this->skuFromOrderItem = $skuFromOrderItem;
    }

    public function aroundExecute(
        \Magento\InventoryShippingAdminUi\Controller\Adminhtml\SourceSelection\Index $subject,
        \Closure $proceed
    ) {
        $orderId = $subject->getRequest()->getParam('order_id');

        if (!$orderId) {
            return $proceed($subject);
        }
        $reservedProducts = $this->reservedProductsRepository->getByOrderId($orderId);

        if (!$this->configProvider->isEnabled()
            || !$this->configProvider->getShippingCostOnSourceEnabled()
            || !$reservedProducts
        ) {
            return $proceed($subject);
        }

        $this->restoreReservedProducts($reservedProducts);
        $result = $this->createShipments($reservedProducts, $orderId);
        $this->reservedProductsRepository->deleteByOrderId($orderId);

        if (!empty($result['success'])) {
            $this->messageManager->addSuccessMessage(__("%1 shipment(s) has been created", $result['success']));
        }

        if (!empty($result['errors'])) {
            $this->messageManager->addWarningMessage(
                __(
                    "Cannot do shipment(s) for source(s) with code %1",
                    implode(", ", $result['errors'])
                )
            );
        }

        $this->redirect->redirect(
            $subject->getResponse(),
            'sales/order/view/order_id/' . $orderId
        );
    }

    /**
     * Split items in order by sources and create shipment for each source
     *
     * @param array $reservedProducts
     * @param string $orderId
     *
     * @return array
     * @throws LocalizedException
     */
    public function createShipments($reservedProducts, $orderId)
    {
        /** @var \Magento\Sales\Model\Order $order */
        $order = $this->orderRepository->get($orderId);

        if (!$order->canShip()) {
            throw new LocalizedException(__('Cannot do shipment for the order'));
        }
        $sourcesWithItems = [];
        $orderItems = $order->getAllItems();
        $orderItemsQtyToShip = [];

        foreach ($reservedProducts as $reservedProduct) {
            if (!isset($sourcesWithItems[$reservedProduct->getSourceCode()])) {
                $sourcesWithItems[$reservedProduct->getSourceCode()] = [];
            }

            $reservedQty = $reservedProduct->getQuantity();

            foreach ($orderItems as $orderItem) {
                if (!isset($orderItemsQtyToShip[$orderItem->getId()])) {
                    $orderItemsQtyToShip[$orderItem->getId()] = $orderItem->getQtyToShip();
                }

                $orderItemSku = $this->skuFromOrderItem->execute($orderItem);

                if ($reservedProduct->getSku() === $orderItemSku
                    && $orderItemsQtyToShip[$orderItem->getId()]
                    && $reservedQty
                ) {
                    $qty = $reservedQty <= $orderItemsQtyToShip[$orderItem->getId()]
                        ? $reservedQty
                        : $orderItemsQtyToShip[$orderItem->getId()];

                    array_push(
                        $sourcesWithItems[$reservedProduct->getSourceCode()],
                        $this->convertOrder->itemToShipmentItem($orderItem)->setQty($qty)
                    );

                    $reservedQty -= $qty;
                    $orderItemsQtyToShip[$orderItem->getId()] -= $qty;
                }
            }
        }

        return $this->createShipmentsBySources($sourcesWithItems, $order);
    }

    /**
     * Create shipments for each source, returns number of success and failures.
     *
     * @param array $sourcesWithItems
     * @param \Magento\Sales\Model\Order $order
     *
     * @return array
     */
    public function createShipmentsBySources($sourcesWithItems, $order)
    {
        $errors = [];
        $success = 0;

        foreach ($sourcesWithItems as $sourceCode => $items) {
            $orderShipment = $this->convertOrder->toShipment($order);

            foreach ($items as $item) {
                $orderShipment->addItem($item);
            }
            $shipmentExtension = $this->shipmentExtensionFactory->create();
            $shipmentExtension->setSourceCode($sourceCode);
            $orderShipment->setExtensionAttributes($shipmentExtension);
            $orderShipment->register();

            try {
                $this->shipmentRepository->save($orderShipment);
                $order = $orderShipment->getOrder()->setStatus(OrderModel::STATE_PROCESSING);
                $this->orderRepository->save($order);
                $success++;
            } catch (\Exception $e) {
                array_push($errors, $sourceCode);
                continue;
            }
        }

        return ['success' => $success, 'errors' => $errors];
    }

    /**
     * Restore reserved products to stock due magento stock availability decrease while shipment creation.
     *
     * @param array $reservedProducts
     */
    public function restoreReservedProducts($reservedProducts)
    {
        $itemsToRestore = [];

        foreach ($reservedProducts as $reservedProduct) {
            foreach ($this->sourceItemsBySku->execute($reservedProduct->getSku()) as $item) {
                if ($item->getSourceCode() === $reservedProduct->getSourceCode()) {
                    $item->setQuantity($item->getQuantity() + $reservedProduct->getQuantity());
                    array_push($itemsToRestore, $item);
                }
            }
        }
        $this->sourceItemsSave->execute($itemsToRestore);
    }
}
