<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Observer;

use Amasty\AdvancedMSI\Model\Config\Source\Email\Template;
use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\CollectionFactory;
use Amasty\AdvancedMSI\Plugin\InventorySourceUi\SourceDataProvider;
use Amasty\AdvancedMSI\Utils\Email;
use Magento\Directory\Model\CountryFactory;
use Magento\Email\Model\ResourceModel\Template as TemplateResource;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Mail\Template\FactoryInterface as TemplateFactory;
use Magento\Framework\Mail\TemplateInterface;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use Magento\InventoryLowQuantityNotification\Model\ResourceModel\LowQuantityCollectionFactory;

class ShipmentPlacer implements ObserverInterface
{
    /**
     * @var Email
     */
    private $emailSender;

    /**
     * @var CollectionFactory
     */
    private $emailCollectionFactory;

    /**
     * @var LowQuantityCollectionFactory
     */
    private $lowQuantityCollectionFactory;

    /**
     * @var SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var CountryFactory
     */
    private $countryFactory;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var TemplateFactory
     */
    private $templateFactory;

    /**
     * @var TemplateResource
     */
    private $templateResource;

    /**
     * @var array
     */
    private $emailParams = [];

    public function __construct(
        Email $emailSender,
        CollectionFactory $emailCollectionFactory,
        LowQuantityCollectionFactory $lowQuantityCollectionFactory,
        SourceRepositoryInterface $sourceRepository,
        CountryFactory $countryFactory,
        ConfigProvider $configProvider,
        TemplateFactory $templateFactory,
        TemplateResource $templateResource
    ) {
        $this->emailSender = $emailSender;
        $this->emailCollectionFactory = $emailCollectionFactory;
        $this->lowQuantityCollectionFactory = $lowQuantityCollectionFactory;
        $this->sourceRepository = $sourceRepository;
        $this->countryFactory = $countryFactory;
        $this->configProvider = $configProvider;
        $this->templateFactory  = $templateFactory;
        $this->templateResource = $templateResource;
    }

    public function execute(Observer $observer)
    {
        $shipment = $observer->getEvent()->getShipment();
        $this->lowStockSend($shipment);
        $sourceCode = $shipment->getExtensionAttributes()->getSourceCode();
        $this->setEmailParams($sourceCode, $shipment);
        $emails = $this->emailCollectionFactory->create()
            ->getEmailsByType($sourceCode, SourceDataProvider::SHIPMENT_EMAILS);

        if (!empty($emails)) {
            $templateId = $this->configProvider->getEmailTemplateId(Template::SHIPMENT_TEMPLATE_ID);
            $template = $this->templateFactory->get($templateId);

            if ($template->getId()) {
                $this->prepareLegacyTemplate($template);
            }

            $this->emailSender->sendEmail(
                $emails,
                0,
                $templateId,
                $this->emailParams
            );
        }
    }

    private function lowStockSend($shipment)
    {
        $sourceCode = $shipment->getExtensionAttributes()->getSourceCode();
        $this->setEmailParams($sourceCode, $shipment);
        $shipmentItems = $shipment->getItems();

        foreach ($shipmentItems as $item) {
            $lowItems = $this->lowQuantityCollectionFactory->create()
                ->addFieldToFilter('sku', $item->getSku())
                ->addFieldToFilter('source_code', $sourceCode);

            foreach ($lowItems as $lowItem) {
                $emails = $this->emailCollectionFactory->create()
                    ->getEmailsByType($sourceCode, SourceDataProvider::LOW_STOCK_EMAILS);

                if (!empty($emails)) {
                    $this->emailParams['product_name'] = $item->getName();
                    $this->emailParams['product_sku'] = $item->getSku();
                    $this->emailParams['available_qty'] = (float)$lowItem->getQuantity();
                    $this->emailSender->sendEmail(
                        $emails,
                        0,
                        $this->configProvider->getEmailTemplateId(Template::LOW_STOCK_TEMPLATE_ID),
                        $this->emailParams
                    );
                }
            }
        }
    }

    private function setEmailParams($sourceCode, $shipment)
    {
        $source = $this->sourceRepository->get($sourceCode);
        $country = $this->countryFactory->create()->loadByCode($source->getCountryId());

        $this->emailParams = [
            'shipment' => $shipment,
            'order' => $shipment->getOrder(),
            'order_id' => $shipment->getOrder()->getIncrementId(),
            'source_country' => $country->getName(),
            'source_region' => $source->getRegion(),
            'source_name' => $source->getName(),
            'source_code' => $source->getSourceCode(),
            'source_city' => $source->getCity(),
            'source_street' => $source->getStreet(),
            'source_postcode' => $source->getPostcode()
        ];
    }

    private function prepareLegacyTemplate(TemplateInterface $template): void
    {
        $this->templateResource->load($template, $template->getId());

        if ($template->getTemplateCode() && !$template->getData('is_legacy')) {
            $template->setData('is_legacy', 1);
            $this->templateResource->save($template);
        }
    }
}
