<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Observer;

use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\Repository\ReservedProductsRepository;
use Amasty\AdvancedMSI\Model\ReservedProducts;
use Amasty\AdvancedMSI\Model\ReservedProductsFactory;
use Amasty\AdvancedMSI\Model\SourceSelection\SourceInfoRegister;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\InventoryApi\Api\GetSourceItemsBySkuInterface;
use Magento\InventoryApi\Api\SourceItemsSaveInterface;
use Magento\Quote\Model\QuoteFactory;

class OrderPlacer implements ObserverInterface
{
    /**
     * @var SourceInfoRegister
     */
    private $sourceInfoRegister;

    /**
     * @var ReservedProductsRepository
     */
    private $reservedProductsRepository;

    /**
     * @var ReservedProducts
     */
    private $reservedProducts;

    /**
     * @var ReservedProductsFactory
     */
    private $reservedProductsFactory;

    /**
     * @var GetSourceItemsBySkuInterface
     */
    private $itemsBySku;

    /**
     * @var SourceItemsSaveInterface
     */
    private $sourceItemsSave;

    /**
     * @var ConfigProvider
     */
    private $configProvider;
    /**
     * @var QuoteFactory
     */
    private $quoteFactory;

    public function __construct(
        SourceInfoRegister $sourceInfoRegister,
        ReservedProductsRepository $reservedProductsRepository,
        ReservedProducts $reservedProducts,
        ReservedProductsFactory $reservedProductsFactory,
        GetSourceItemsBySkuInterface $itemsBySku,
        SourceItemsSaveInterface $sourceItemsSave,
        ConfigProvider $configProvider,
        QuoteFactory $quoteFactory
    ) {
        $this->sourceInfoRegister = $sourceInfoRegister;
        $this->reservedProductsRepository = $reservedProductsRepository;
        $this->reservedProducts = $reservedProducts;
        $this->reservedProductsFactory = $reservedProductsFactory;
        $this->itemsBySku = $itemsBySku;
        $this->sourceItemsSave = $sourceItemsSave;
        $this->configProvider = $configProvider;
        $this->quoteFactory = $quoteFactory;
    }

    public function execute(Observer $observer)
    {
        $order = $observer->getEvent()->getOrder();
        $orderId = (int)$order->getId();
        if (!$this->sourceInfoRegister->getSourceSelectionResult()) {
            $this->quoteFactory->create()
                ->loadByIdWithoutStore($order->getQuoteId())
                ->getShippingAddress()
                ->requestShippingRates();
        }
        $productsData = $this->sourceInfoRegister->getReservedProductsFromSourceSelection();

        if (!$this->configProvider->isEnabled()
            || !$this->configProvider->getShippingCostOnSourceEnabled()
            || !$orderId
            || !$productsData
        ) {
            return;
        }
        $itemsToSave = [];

        foreach ($productsData as $data) {
            $reservedProduct = $this->reservedProductsFactory->create();
            $reservedProduct->setOrderId($orderId);
            $reservedProduct->setSku($data['sku']);
            $reservedProduct->setQuantity($data['quantity']);
            $reservedProduct->setSourceCode($data['source_code']);

            foreach ($this->itemsBySku->execute($data['sku']) as $item) {
                if ($item->getSourceCode() === $data['source_code']) {
                    $item->setQuantity($item->getQuantity() - $data['quantity']);
                    array_push($itemsToSave, $item);
                }
            }

            $this->reservedProductsRepository->save($reservedProduct);
        }
        $this->sourceItemsSave->execute($itemsToSave);
    }
}
