<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection;

use Magento\Store\Model\StoreManagerInterface;
use Magento\InventoryApi\Api\Data\StockInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventorySalesApi\Api\StockResolverInterface;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySourceSelection\Model\Result\SourceSelectionResult;
use Magento\Inventory\Model\ResourceModel\StockSourceLink\CollectionFactory as StockSourceLinkCollectionFactory;

class SourceInfoRegister
{
    /**
     * @var int|null
     */
    protected $stockId = null;

    /**
     * @var SourceSelectionResult
     */
    private $sourceSelectionResult;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var StockResolverInterface
     */
    private $stockResolver;

    /**
     * @var StockSourceLinkCollectionFactory
     */
    private $stockSourceLinkCollectionFactory;

    public function __construct(
        StoreManagerInterface $storeManager,
        StockResolverInterface $stockResolver,
        StockSourceLinkCollectionFactory $stockSourceLinkCollectionFactory
    ) {
        $this->storeManager = $storeManager;
        $this->stockResolver = $stockResolver;
        $this->stockSourceLinkCollectionFactory = $stockSourceLinkCollectionFactory;
    }

    public function setSourceSelectionResult(SourceSelectionResult $sourceSelectionResult): void
    {
        $this->sourceSelectionResult = $sourceSelectionResult;
    }

    public function getSourceSelectionResult(): ?SourceSelectionResult
    {
        return $this->sourceSelectionResult;
    }

    public function getReservedProductsFromSourceSelection(): ?array
    {
        $sourceSelectionResult = $this->sourceSelectionResult;
        $reservedProducts = [];
        if ($sourceSelectionResult && $this->canReserve()) {
            foreach ($sourceSelectionResult->getSourceSelectionItems() as $item) {
                if ($item->getQtyToDeduct() != 0) {
                    $productInfo = [
                        'sku' => $item->getSku(),
                        'quantity' => $item->getQtyToDeduct(),
                        'source_code' => $item->getSourceCode()
                    ];
                    array_push($reservedProducts, $productInfo);
                }
            }
        }

        return $reservedProducts;
    }

    /**
     * Check that more than one source is assigned for the current stock
     */
    private function canReserve(): bool
    {
        $stockSourceLinkSize = 0;

        if ($stockId = $this->getStockId()) {
            $stockSourceLinkSize = $this->stockSourceLinkCollectionFactory->create()
                ->addFieldToFilter(StockInterface::STOCK_ID, $stockId)
                ->getSize();
        }

        return $stockSourceLinkSize > 1;
    }

    private function getStockId(): ?int
    {
        if (!$this->stockId) {
            try {
                $stock = $this->stockResolver->execute(
                    SalesChannelInterface::TYPE_WEBSITE,
                    $this->storeManager->getWebsite()->getCode()
                );
            } catch (LocalizedException $e) {
                $stock = null;
            }

            $this->stockId = $stock ? (int)$stock->getStockId() : null;
        }

        return $this->stockId;
    }
}
