<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection;

use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use Magento\InventoryDistanceBasedSourceSelection\Model\DistanceProvider\GetDistanceFromSourceToAddress;
use Magento\InventorySourceSelection\Model\AddressFactory;
use Magento\Sales\Model\OrderRepository;

/**
 * Service calculating distance between Sources and Customer for distance based algorithm used in combine algorithm
 */
class GetDistancesBetweenSourcesAndCustomer
{
    /**
     * @var GetDistanceFromSourceToAddress
     */
    private $getDistanceFromSourceToAddress;

    /**
     * @var SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var CustomerAddressRegister
     */
    private $addressRegister;

    /**
     * @var AddressFactory
     */
    private $addressFactory;

    /**
     * @var OrderRepository
     */
    private $orderRepository;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * GetDistancesBetweenSourcesAndCustomer constructor.
     *
     * @param GetDistanceFromSourceToAddress $getDistanceFromSourceToAddress
     * @param SourceRepositoryInterface $sourceRepository
     * @param CustomerAddressRegister $addressRegister
     * @param AddressFactory $addressFactory
     */
    public function __construct(
        GetDistanceFromSourceToAddress $getDistanceFromSourceToAddress,
        SourceRepositoryInterface $sourceRepository,
        CustomerAddressRegister $addressRegister,
        AddressFactory $addressFactory,
        OrderRepository $orderRepository,
        RequestInterface $request
    ) {
        $this->getDistanceFromSourceToAddress = $getDistanceFromSourceToAddress;
        $this->sourceRepository = $sourceRepository;
        $this->addressRegister = $addressRegister;
        $this->addressFactory = $addressFactory;
        $this->orderRepository = $orderRepository;
        $this->request = $request;
    }

    /**
     * Calculate distance from source addresses to customer
     *
     * @param array $stockSourceLinks
     *
     * @return array
     * @throws LocalizedException
     */
    public function execute(array $stockSourceLinks): array
    {
        if ($destinationAddress = $this->addressRegister->getDestinationAddress()) {
            $address = $this->addressFactory->create($destinationAddress);
        } else {
            $orderId = $this->request->getParam('order_id') ?: $this->request->getParam('orderId');
            /** @var \Magento\Sales\Model\Order\Address $shippingAddress */
            $shippingAddress = $this->orderRepository->get($orderId)->getShippingAddress();

            $address = $this->addressFactory->create(
                [
                    'country' => (string)$shippingAddress->getCountryId(),
                    'postcode' => (string)$shippingAddress->getPostcode(),
                    'street' => (string)$shippingAddress->getStreetLine(1),
                    'region' => (string)$shippingAddress->getRegion(),
                    'city' => (string)$shippingAddress->getCity()
                ]
            );
        }
        $sources = $this->prepareSources($stockSourceLinks);
        $preparedSources = $sourcesWithoutDistance = [];

        /** @var SourceInterface $source */
        foreach ($sources as $source) {
            try {
                $distance = $this->getDistanceFromSourceToAddress->execute(
                    $source,
                    $address
                );
                $preparedSources[$source->getSourceCode()] = [
                    'source' => $source,
                    'distance' => $distance
                ];
            } catch (LocalizedException $e) {
                $sourcesWithoutDistance[$source->getSourceCode()] = [
                    'source' => $source,
                    'distance' => 0
                ];
            }
        }

        return array_merge($preparedSources, $sourcesWithoutDistance);
    }

    /**
     * Get source objects from stock source links
     *
     * @param array $stockSourceLinks
     *
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function prepareSources(array $stockSourceLinks): array
    {
        $sources = [];

        /** @var SourceInterface $source */
        foreach ($stockSourceLinks as $source) {
            $source = $this->sourceRepository->get($source->getSourceCode());
            if ($source->isEnabled()) {
                $sources[$source->getSourceCode()] = $source;
            }
        }

        return $sources;
    }
}
