<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection;

class CombinedAlgorithmRegistry
{
    /**
     * @var CombinedAlgorithmData[]
     */
    private $sourceSelectionAlgorithms;

    /**
     * @param CombinedAlgorithmData[] $sourceSelectionAlgorithms
     */
    public function __construct(
        array $sourceSelectionAlgorithms = []
    ) {
        $this->sourceSelectionAlgorithms = $this->getSortedAlgorithms($sourceSelectionAlgorithms);
    }

    /**
     * Return all available algorithms for combined method sorted by priority
     *
     * @return CombinedAlgorithmData[]
     */
    public function getAll()
    {
        return $this->sourceSelectionAlgorithms;
    }

    /**
     * @param CombinedAlgorithmData[] $sourceSelectionAlgorithms
     *
     * @return CombinedAlgorithmData[]
     */
    private function getSortedAlgorithms(array $sourceSelectionAlgorithms): array
    {
        uasort($sourceSelectionAlgorithms, function ($a, $b) {
            /** @var CombinedAlgorithmData $a */
            /** @var CombinedAlgorithmData $b */
            if ($a->getPriority() == $b->getPriority()) {
                return 0;
            }

            return ($a->getPriority() < $b->getPriority()) ? -1 : 1;
        });

        return $sourceSelectionAlgorithms;
    }

    /**
     * @return CombinedAlgorithmData[]
     */
    public function getActive(): array
    {
        $allAlgorithms = $this->getAll();
        foreach ($allAlgorithms as $key => $algorithm) {
            if (!$algorithm->getIsActive()) {
                unset($allAlgorithms[$key]);
            }
        }

        return $allAlgorithms;
    }
}
