<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection\Algorithms;

use Amasty\AdvancedMSI\Model\SourceSelection\GetSourcesWithFullStockByRequest;
use Amasty\AdvancedMSI\Model\SourceItem\GetSourceItemsSortedByStock;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use \Magento\InventorySourceSelectionApi\Model\SourceSelectionInterface;

/**
 * {@inheritdoc}
 * This shipping algorithm iterates over all the sources one by one in stock order and tries to ship from one Source
 */
class StockBased implements SourceSelectionInterface
{
    /**
     * Algorithm code
     */
    public const CODE = 'stock';

    /**
     * @var GetSourcesAssignedToStockOrderedByPriorityInterface
     */
    private $getSourcesAssignedToStockOrderedByPriority;

    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;

    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;

    /**
     * @var GetSourcesWithFullStockByRequest
     */
    private $sourcesWithFullStockByRequest;

    /**
     * @var GetSourceItemsSortedByStock
     */
    private $getSourceItemsSortedByStock;

    /**
     * PrioritySourceSelectionAlgorithm constructor.
     *
     * @param GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority
     * @param SourceSelectionItemInterfaceFactory                 $sourceSelectionItemFactory
     * @param SourceSelectionResultInterfaceFactory               $sourceSelectionResultFactory
     * @param GetSourcesWithFullStockByRequest                    $sourcesWithFullStockByRequest
     * @param GetSourceItemsSortedByStock                         $getSourceItemsSortedByStock
     */
    public function __construct(
        GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        GetSourcesWithFullStockByRequest $sourcesWithFullStockByRequest,
        GetSourceItemsSortedByStock $getSourceItemsSortedByStock
    ) {
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->sourcesWithFullStockByRequest = $sourcesWithFullStockByRequest;
        $this->getSourceItemsSortedByStock = $getSourceItemsSortedByStock;
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     *
     * @return SourceSelectionResultInterface
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        $isShippable = true;
        $sourceToDeduct = false;
        $sourceItemSelections = $sourceCodes = [];
        $sources = $this->getEnabledSourcesOrderedByPriorityByStockId($inventoryRequest->getStockId());
        foreach ($sources as $source) {
            $sourceCodes[] = $source->getSourceCode();
        }
        $sourcesWithFullStock = $this->sourcesWithFullStockByRequest->execute($sourceCodes, $inventoryRequest);
        if (!empty($sourcesWithFullStock)) {
            $sourceToDeduct = current($sourcesWithFullStock);
        }

        foreach ($inventoryRequest->getItems() as $item) {
            $itemSku = $item->getSku();
            $qtyToDeliver = $item->getQty();

            $sourceItems = $this->getSourceItemsSortedByStock->execute($sourceCodes, $itemSku);

            foreach ($sourceItems as $sourceItem) {

                $sourceItemQty = $sourceItem->getQuantity();
                // check if source has some qty of SKU, so it's possible to take them into account
                if ($this->isZero((float)$sourceItemQty)) {
                    continue;
                }
                $qtyToDeduct = 0;
                if (!$sourceToDeduct || $sourceToDeduct == $sourceItem->getSourceCode()) {
                    $qtyToDeduct = min($sourceItemQty, $qtyToDeliver);
                }

                $sourceItemSelections[] = $this->sourceSelectionItemFactory->create(
                    [
                        'sourceCode'   => $sourceItem->getSourceCode(),
                        'sku'          => $itemSku,
                        'qtyToDeduct'  => $qtyToDeduct,
                        'qtyAvailable' => $sourceItemQty
                    ]
                );

                $qtyToDeliver -= $qtyToDeduct;
            }

            // if we go throw all sources from the stock and there is still some qty to delivery,
            // then it doesn't have enough items to delivery
            if (!$this->isZero($qtyToDeliver)) {
                $isShippable = false;
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $sourceItemSelections,
                'isShippable'          => $isShippable
            ]
        );
    }

    /**
     * Compare float number with some epsilon
     *
     * @param float $floatNumber
     *
     * @return bool
     */
    private function isZero(float $floatNumber): bool
    {
        return $floatNumber < 0.0000001;
    }

    /**
     * Get enabled sources ordered by priority by $stockId
     *
     * @param int $stockId
     *
     * @return array
     */
    private function getEnabledSourcesOrderedByPriorityByStockId(int $stockId): array
    {
        $sources = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);
        $sources = array_filter(
            $sources,
            function (SourceInterface $source) {
                return $source->isEnabled();
            }
        );

        return $sources;
    }
}
