<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;

use Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Amasty\AdvancedMSI\Model\SourceSelection\GetSourcesWithFullStockByRequest;

/**
 * Part of combined algorithm, which modify equal list, left only sources with full stock
 */
class StockBased
{
    /**
     * @var GetSourcesWithFullStockByRequest
     */
    private $sourcesWithFullStockByRequest;

    /**
     * @var EqualSourcesRegister
     */
    private $sourcesRegister;

    /**
     * StockBased constructor.
     *
     * @param GetSourcesWithFullStockByRequest $sourcesWithFullStockByRequest
     * @param EqualSourcesRegister $sourcesRegister
     */
    public function __construct(
        GetSourcesWithFullStockByRequest $sourcesWithFullStockByRequest,
        EqualSourcesRegister $sourcesRegister
    ) {
        $this->sourcesWithFullStockByRequest = $sourcesWithFullStockByRequest;
        $this->sourcesRegister = $sourcesRegister;
    }

    /**
     * Find sources with full stock. Add them to equal list
     *
     * @param InventoryRequestInterface $inventoryRequest
     */
    public function execute(InventoryRequestInterface $inventoryRequest): void
    {
        $sources = $this->sourcesRegister->getEqual();
        $sourceCodes = [];

        foreach ($sources as $source) {
            $sourceCodes[] = $source->getSourceCode();
        }

        $fullStockSources = $this->sourcesWithFullStockByRequest->execute($sourceCodes, $inventoryRequest);

        if (!empty($fullStockSources)) {
            $equal = $this->sourcesRegister->getEqual();
            $rest = $this->sourcesRegister->getRest();

            foreach ($equal as $key => $source) {
                if (!in_array($source->getSourceCode(), $fullStockSources)) {
                    unset($equal[$key]);
                    $rest[] = $source;
                }
            }

            $this->sourcesRegister->setEqual($equal);
            $this->sourcesRegister->setRest($rest);
        }
    }
}
