<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;

use Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\Inventory\Model\Source\Command\GetSourcesAssignedToStockOrderedByPriority;

/**
 * Part of combined algorithm, which sorted equal list by priority
 */
class PriorityBased
{
    /**
     * @var GetSourcesAssignedToStockOrderedByPriority
     */
    private $getSourcesAssignedToStockOrderedByPriority;

    /**
     * @var EqualSourcesRegister
     */
    private $sourcesRegister;

    /**
     * PriorityBased constructor.
     *
     * @param GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority
     * @param EqualSourcesRegister $sourcesRegister
     */
    public function __construct(
        GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority,
        EqualSourcesRegister $sourcesRegister
    ) {
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->sourcesRegister = $sourcesRegister;
    }

    /**
     * Sort equal list by priority
     *
     * @param InventoryRequestInterface $inventoryRequest
     *
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(InventoryRequestInterface $inventoryRequest): void
    {
        $sourceCodes = $sortedSources = [];
        $sources = $this->getSourcesAssignedToStockOrderedByPriority->execute($inventoryRequest->getStockId());

        $equal = $this->sourcesRegister->getEqual();
        foreach ($equal as $source) {
            $sourceCodes[] = $source->getSourceCode();
        }

        foreach ($sources as $source) {
            if (in_array($source->getSourceCode(), $sourceCodes)) {
                $sortedSources[] = $source;
            }
        }

        $this->sourcesRegister->setEqual($sortedSources);
    }
}
