<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined;

use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\SourceSelection\GetDistancesBetweenSourcesAndCustomer;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;

/**
 * Part of combined algorithm, which modify equal list,
 * left only nearest source and sources which are not father than round distance value
 */
class DistanceBased
{
    /**
     * @var GetDistancesBetweenSourcesAndCustomer
     */
    private $getDistancesBetweenSourcesAndCustomer;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var EqualSourcesRegister
     */
    private $sourcesRegister;

    /**
     * DistanceBased constructor.
     *
     * @param GetDistancesBetweenSourcesAndCustomer $getDistancesBetweenSourcesAndCustomer
     * @param ConfigProvider $configProvider
     * @param EqualSourcesRegister $sourcesRegister
     */
    public function __construct(
        GetDistancesBetweenSourcesAndCustomer $getDistancesBetweenSourcesAndCustomer,
        ConfigProvider $configProvider,
        EqualSourcesRegister $sourcesRegister
    ) {
        $this->getDistancesBetweenSourcesAndCustomer = $getDistancesBetweenSourcesAndCustomer;
        $this->configProvider = $configProvider;
        $this->sourcesRegister = $sourcesRegister;
    }

    /**
     * Find nearest source and add sources which are not father then RoundeDistance value
     * Add them to equal list
     *
     * @param InventoryRequestInterface $inventoryRequest
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(InventoryRequestInterface $inventoryRequest): void
    {
        $sources = $this->getDistancesBetweenSourcesAndCustomer->execute($this->sourcesRegister->getEqual());
        $roundDistance = $this->configProvider->getRoundDistance();

        usort($sources, function ($a, $b) {
            if ($a['distance'] == $b['distance']) {
                return 0;
            }
            return ($a['distance'] < $b['distance']) ? -1 : 1;
        });

        //to avoid error while creating invoice for downloadable product
        if (empty($sources)) {
            $this->sourcesRegister->setEqual([]);
            $this->sourcesRegister->setRest([]);

            return;
        }
        $distance = $sources[0]['distance'] + $roundDistance;
        $eqSources = [];
        $restSources = [];

        foreach ($sources as $source) {
            if ($source['distance'] <= $distance) {
                $eqSources[] = $source['source'];
            } else {
                $restSources[] = $source['source'];
            }
        }

        $this->sourcesRegister->setEqual($eqSources);
        $this->sourcesRegister->setRest(array_merge($this->sourcesRegister->getRest(), $restSources));
    }
}
