<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\SourceSelection\Algorithms;

use Amasty\AdvancedMSI\Model\SourceSelection\Algorithms\Combined\EqualSourcesRegister;
use Magento\CatalogInventory\Model\Stock;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventoryConfiguration\Model\GetStockItemConfiguration;
use Magento\InventorySourceSelectionApi\Model\SourceSelectionInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Amasty\AdvancedMSI\Model\ConfigProvider;
use Amasty\AdvancedMSI\Model\SourceSelection\CombinedAlgorithmRegistry;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\InventoryApi\Api\GetStockSourceLinksInterface;
use Magento\InventoryApi\Api\Data\StockSourceLinkInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemRepositoryInterface;
use Magento\Framework\ObjectManagerInterface;

/**
 * {@inheritdoc}
 * This shipping algorithm use data from all other algorithms, selected by admin
 */
class Combined implements SourceSelectionInterface
{
    public const CODE = 'combined';

    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var GetStockSourceLinksInterface
     */
    private $getStockSourceLinks;

    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;

    /**
     * @var SourceItemRepositoryInterface
     */
    private $sourceItemRepository;

    /**
     * @var CombinedAlgorithmRegistry
     */
    private $algorithmRegistry;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var EqualSourcesRegister
     */
    private $sourcesRegister;

    /**
     * @var GetStockItemConfiguration
     */
    private $getStockItemConfiguration;

    public function __construct(
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        ConfigProvider $configProvider,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        GetStockSourceLinksInterface $getStockSourceLinks,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        SourceItemRepositoryInterface $sourceItemRepository,
        CombinedAlgorithmRegistry $algorithmRegistry,
        ObjectManagerInterface $objectManager,
        EqualSourcesRegister $sourcesRegister,
        GetStockItemConfiguration $getStockItemConfiguration
    ) {
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->getStockSourceLinks = $getStockSourceLinks;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->sourceItemRepository = $sourceItemRepository;
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->configProvider = $configProvider;
        $this->algorithmRegistry = $algorithmRegistry;
        $this->objectManager = $objectManager;
        $this->sourcesRegister = $sourcesRegister;
        $this->getStockItemConfiguration = $getStockItemConfiguration;
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     *
     * @return SourceSelectionResultInterface
     *
     * @throws LocalizedException
     * @throws InputException
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        $isShippable = true;
        $sourceItemSelections = [];
        $this->prepareEqualList($inventoryRequest);

        // all sources should be used for next step. But with a custom sorting
        $sources = $this->sourcesRegister->getAll();

        foreach ($inventoryRequest->getItems() as $item) {
            $itemSku = $item->getSku();
            $qtyToDeliver = $item->getQty();
            $stockItem = $this->getStockItemConfiguration->execute($itemSku, $inventoryRequest->getStockId());

            foreach ($sources as $source) {
                $sourceItem = $this->getSourceItemBySourceCodeAndSku($source->getSourceCode(), $itemSku);
                if (null === $sourceItem) {
                    continue;
                }

                if ($sourceItem->getStatus() != SourceItemInterface::STATUS_IN_STOCK) {
                    continue;
                }

                $sourceItemQty = $sourceItem->getQuantity();
                $qtyToDeduct = min($sourceItemQty, $qtyToDeliver);

                // check if source has some qty of SKU, so it's possible to take them into account
                if ($this->isZero((float)$sourceItemQty)) {
                    if ($stockItem->isManageStock()
                        && !in_array(
                            $stockItem->getBackorders(),
                            [Stock::BACKORDERS_YES_NONOTIFY, Stock::BACKORDERS_YES_NOTIFY]
                        )
                    ) {
                        continue;
                    }

                    $qtyToDeduct = $qtyToDeliver;
                }

                $sourceItemSelections[] = $this->sourceSelectionItemFactory->create([
                    'sourceCode' => $sourceItem->getSourceCode(),
                    'sku' => $itemSku,
                    'qtyToDeduct' => $qtyToDeduct,
                    'qtyAvailable' => $sourceItemQty
                ]);

                $qtyToDeliver -= $qtyToDeduct;
            }

            // if we go throw all sources from the stock and there is still some qty to delivery,
            // then it doesn't have enough items to delivery
            if (!$this->isZero($qtyToDeliver)) {
                $isShippable = false;
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $sourceItemSelections,
                'isShippable' => $isShippable
            ]
        );
    }

    /**
     * This method calls all other algorithms to prepare list of equal sources
     *
     * @param InventoryRequestInterface $inventoryRequest
     *
     * @throws LocalizedException
     * @throws InputException
     */
    private function prepareEqualList(InventoryRequestInterface $inventoryRequest): void
    {
        $this->sourcesRegister->setEqual($this->getStockSourceLinks($inventoryRequest->getStockId()));
        /** @var \Amasty\AdvancedMSI\Model\SourceSelection\CombinedAlgorithmData[] $algorithms */
        $algorithms = $this->algorithmRegistry->getActive();
        foreach ($algorithms as $algorithm) {
            // One element in equal list should stopped combined algorithm
            if (count($this->sourcesRegister->getEqual()) == 1) {
                break;
            }
            if ($algorithm->getIsActive()) {
                $object = $this->objectManager->get($algorithm->getClass());
                $object->execute($inventoryRequest);
                if ($algorithm->getIsShouldStop()) {
                    return;
                }
            }
        }
    }

    /**
     * Get all stock-source links by given stockId
     *
     * @param int $stockId
     *
     * @return array
     */
    private function getStockSourceLinks(int $stockId): array
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(StockSourceLinkInterface::STOCK_ID, $stockId)
            ->create();
        $searchResult = $this->getStockSourceLinks->execute($searchCriteria);

        return $searchResult->getItems();
    }

    /**
     * Compare float number with some epsilon
     *
     * @param float $floatNumber
     *
     * @return bool
     */
    private function isZero(float $floatNumber): bool
    {
        return $floatNumber < 0.0000001;
    }

    /**
     * Returns source item from specific source by given SKU. Return null if source item is not found
     *
     * @param string $sourceCode
     * @param string $sku
     *
     * @return SourceItemInterface|null
     */
    private function getSourceItemBySourceCodeAndSku(string $sourceCode, string $sku)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SourceItemInterface::SOURCE_CODE, $sourceCode)
            ->addFilter(SourceItemInterface::SKU, $sku)
            ->create();
        $sourceItemsResult = $this->sourceItemRepository->getList($searchCriteria);

        return $sourceItemsResult->getTotalCount() > 0 ? current($sourceItemsResult->getItems()) : null;
    }
}
