<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\Source\DataProvider;

use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\RequestInterface;
use Magento\CatalogInventory\Api\StockConfigurationInterface;

class ProductListing extends \Magento\Catalog\Ui\DataProvider\Product\ProductDataProvider
{
    public const SOURCE_PRODUCTS_LISTING = 'ammsi_source_product_listing_data_source';
    public const SOURCE_ADD_PRODUCTS_LISTING = 'ammsi_source_add_product_listing_data_source';

    /**
     * @var StockConfigurationInterface
     */
    private $stockConfiguration;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        RequestInterface $request,
        $name,
        $primaryFieldName,
        $requestFieldName,
        CollectionFactory $collectionFactory,
        StockConfigurationInterface $stockConfiguration,
        $addFieldStrategies = [],
        $addFilterStrategies = [],
        array $meta = [],
        array $data = []
    ) {
        parent::__construct(
            $name,
            $primaryFieldName,
            $requestFieldName,
            $collectionFactory,
            $addFieldStrategies,
            $addFilterStrategies,
            $meta,
            $data
        );
        $allowedProductTypes = array_keys(array_filter($stockConfiguration->getIsQtyTypeIds()));
        $this->collection->addAttributeToFilter('type_id', ['in' => $allowedProductTypes]);
        $this->collection->addAttributeToSelect(['status', 'thumbnail', 'name', 'price'], 'left');
        if ($sourceCode = $request->getParam('source_code')) {
            //phpcs:disable
            switch ($name) {
                case self::SOURCE_PRODUCTS_LISTING:
                    $defaultQty = $scopeConfig->getValue('cataloginventory/item_options/notify_stock_qty');
                    $this->collection->getSelect()->joinInner(
                        ['is' => $this->collection->getTable('inventory_source_item')],
                        'e.sku = is.sku',
                        ['is.quantity', 'is.status as source_item_status']
                    )->joinLeft(
                        ['ls' => $this->collection->getTable('inventory_low_stock_notification_configuration')],
                        'is.sku = ls.sku and is.source_code = ls.source_code',
                        [
                            'IFNULL(ls.notify_stock_qty, ' . (float)$defaultQty . ') as notify_stock_qty',
                            'ls.notify_stock_qty as initial_notify_stock_qty'
                        ]
                    )->where('is.source_code = ?', $sourceCode);
                    break;
                case self::SOURCE_ADD_PRODUCTS_LISTING:
                    $this->collection->getSelect()->distinct(true);
                    $this->collection->getSelect()->joinInner(
                        ['is' => $this->collection->getTable('inventory_source_item')],
                        'e.sku = is.sku',
                        []
                    )->where(
                        new \Zend_Db_Expr(
                            'is.sku not in (select sku from ' .
                            $this->collection->getTable('inventory_source_item')
                            . ' where source_code = ?)'
                        ),
                        $sourceCode
                    );
                    break;
                default:
                    $this->collection->setPageSize(0);
            }
            //phpcs:enable
        } else {
            $this->collection->setPageSize(0);
        }
    }

    /**
     * @inheritdoc
     */
    public function addFilter(\Magento\Framework\Api\Filter $filter)
    {
        $filterField = $filter->getField();
        if (self::SOURCE_PRODUCTS_LISTING
            && in_array($filterField, ['notify_stock_qty', 'quantity', 'source_item_status'])
        ) {
            if ($filterField === 'quantity') {
                $filterField = 'is.quantity';
            } elseif ($filterField === 'source_item_status') {
                $filterField = 'is.status';
            } elseif ($filterField === 'notify_stock_qty') {
                $filterField = 'ls.notify_stock_qty';
            }
            switch ($filter->getConditionType()) {
                case 'lteq':
                    $this->getCollection()->getSelect()->where(
                        $filterField . ' <= ?',
                        $filter->getValue()
                    );
                    break;
                case 'gteq':
                    $this->getCollection()->getSelect()->where(
                        $filterField . ' >= ?',
                        $filter->getValue()
                    );
                    break;
                case 'eq':
                    $this->getCollection()->getSelect()->where(
                        $filterField . ' = ?',
                        $filter->getValue()
                    );
                    break;
            }
        } else {
            parent::addFilter($filter);
        }
    }

    public function addOrder($field, $direction)
    {
        if (self::SOURCE_PRODUCTS_LISTING
            && in_array($field, ['notify_stock_qty', 'quantity', 'source_item_status'])
        ) {
            if ($field === 'quantity') {
                $field = 'is.quantity';
            } elseif ($field === 'source_item_status') {
                $field = 'is.status';
            } elseif ($field === 'notify_stock_qty') {
                $field = 'ls.notify_stock_qty';
            }

            $this->getCollection()->getSelect()->order($field . ' ' . $direction);
        } else {
            parent::addOrder($field, $direction);
        }
    }

    public function getData()
    {
        $data = parent::getData();

        if ($this->name == self::SOURCE_PRODUCTS_LISTING && !empty($data['items'])) {
            foreach ($data['items'] as &$item) {
                foreach (['notify_stock_qty', 'quantity', 'initial_notify_stock_qty'] as $field) {
                    if (isset($item[$field])) {
                        $item[$field] = (float)$item[$field];
                    }
                }
            }
        }

        return $data;
    }
}
