<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\Repository;

use Amasty\AdvancedMSI\Api\Data\SourceEmailInterface;
use Amasty\AdvancedMSI\Api\SourceEmailRepositoryInterface;
use Amasty\AdvancedMSI\Model\SourceEmailFactory;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail as SourceEmailResource;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\CollectionFactory;
use Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\Collection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;
use Magento\Framework\Api\SortOrder;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class SourceEmailRepository implements SourceEmailRepositoryInterface
{
    /**
     * @var BookmarkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var SourceEmailFactory
     */
    private $sourceEmailFactory;

    /**
     * @var SourceEmailResource
     */
    private $sourceEmailResource;

    /**
     * Model data storage
     *
     * @var array
     */
    private $sourceEmails;

    /**
     * @var CollectionFactory
     */
    private $sourceEmailCollectionFactory;

    public function __construct(
        BookmarkSearchResultsInterfaceFactory $searchResultsFactory,
        SourceEmailFactory $sourceEmailFactory,
        SourceEmailResource $sourceEmailResource,
        CollectionFactory $sourceEmailCollectionFactory
    ) {
        $this->searchResultsFactory = $searchResultsFactory;
        $this->sourceEmailFactory = $sourceEmailFactory;
        $this->sourceEmailResource = $sourceEmailResource;
        $this->sourceEmailCollectionFactory = $sourceEmailCollectionFactory;
    }

    /**
     * @inheritdoc
     */
    public function save(SourceEmailInterface $sourceEmail)
    {
        try {
            if ($sourceEmail->getSourceEmailId()) {
                $sourceEmail = $this->getById($sourceEmail->getSourceEmailId())->addData($sourceEmail->getData());
            }
            $this->sourceEmailResource->save($sourceEmail);
            unset($this->sourceEmails[$sourceEmail->getSourceEmailId()]);
        } catch (\Exception $e) {
            if ($sourceEmail->getSourceEmailId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save sourceEmail with ID %1. Error: %2',
                        [$sourceEmail->getSourceEmailId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new sourceEmail. Error: %1', $e->getMessage()));
        }

        return $sourceEmail;
    }

    /**
     * @inheritdoc
     */
    public function getById($sourceEmailId)
    {
        if (!isset($this->sourceEmails[$sourceEmailId])) {
            /** @var \Amasty\AdvancedMSI\Model\SourceEmail $sourceEmail */
            $sourceEmail = $this->sourceEmailFactory->create();
            $this->sourceEmailResource->load($sourceEmail, $sourceEmailId);
            if (!$sourceEmail->getSourceEmailId()) {
                throw new NoSuchEntityException(__('SourceEmail with specified ID "%1" not found.', $sourceEmailId));
            }
            $this->sourceEmails[$sourceEmailId] = $sourceEmail;
        }

        return $this->sourceEmails[$sourceEmailId];
    }

    /**
     * @inheritdoc
     */
    public function delete(SourceEmailInterface $sourceEmail)
    {
        try {
            $this->sourceEmailResource->delete($sourceEmail);
            unset($this->sourceEmails[$sourceEmail->getSourceEmailId()]);
        } catch (\Exception $e) {
            if ($sourceEmail->getSourceEmailId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove sourceEmail with ID %1. Error: %2',
                        [$sourceEmail->getSourceEmailId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove sourceEmail. Error: %1', $e->getMessage()));
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteById($sourceEmailId)
    {
        $sourceEmailModel = $this->getById($sourceEmailId);
        $this->delete($sourceEmailModel);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);

        /** @var \Amasty\AdvancedMSI\Model\ResourceModel\SourceEmail\Collection $sourceEmailCollection */
        $sourceEmailCollection = $this->sourceEmailCollectionFactory->create();
        
        // Add filters from root filter group to the collection
        foreach ($searchCriteria->getFilterGroups() as $group) {
            $this->addFilterGroupToCollection($group, $sourceEmailCollection);
        }
        
        $searchResults->setTotalCount($sourceEmailCollection->getSize());
        $sortOrders = $searchCriteria->getSortOrders();
        
        if ($sortOrders) {
            $this->addOrderToCollection($sortOrders, $sourceEmailCollection);
        }
        
        $sourceEmailCollection->setCurPage($searchCriteria->getCurrentPage());
        $sourceEmailCollection->setPageSize($searchCriteria->getPageSize());
        
        $sourceEmails = [];
        /** @var SourceEmailInterface $sourceEmail */
        foreach ($sourceEmailCollection->getItems() as $sourceEmail) {
            $sourceEmails[] = $this->getById($sourceEmail->getSourceEmailId());
        }
        
        $searchResults->setItems($sourceEmails);

        return $searchResults;
    }

    /**
     * Helper function that adds a FilterGroup to the collection.
     *
     * @param FilterGroup $filterGroup
     * @param Collection  $sourceEmailCollection
     *
     * @return void
     */
    private function addFilterGroupToCollection(FilterGroup $filterGroup, Collection $sourceEmailCollection)
    {
        foreach ($filterGroup->getFilters() as $filter) {
            $condition = $filter->getConditionType() ?: 'eq';
            $sourceEmailCollection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
        }
    }

    /**
     * Helper function that adds a SortOrder to the collection.
     *
     * @param SortOrder[] $sortOrders
     * @param Collection  $sourceEmailCollection
     *
     * @return void
     */
    private function addOrderToCollection($sortOrders, Collection $sourceEmailCollection)
    {
        /** @var SortOrder $sortOrder */
        foreach ($sortOrders as $sortOrder) {
            $field = $sortOrder->getField();
            $sourceEmailCollection->addOrder(
                $field,
                ($sortOrder->getDirection() == SortOrder::SORT_DESC) ? SortOrder::SORT_DESC : SortOrder::SORT_ASC
            );
        }
    }

    /**
     * @return \Amasty\AdvancedMSI\Model\SourceEmail
     */
    public function getEmptySourceEmailModel()
    {
        return $this->sourceEmailFactory->create();
    }
}
