<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\Repository;

use Amasty\AdvancedMSI\Api\Data\SourceCustomShippingRateInterface;
use Amasty\AdvancedMSI\Api\SourceCustomShippingRateRepositoryInterface;
use Magento\Framework\Exception\CouldNotSaveException;

class SourceCustomShippingRateRepository implements SourceCustomShippingRateRepositoryInterface
{
    /**
     * @var \Amasty\AdvancedMSI\Model\SourceCustomShippingRateFactory
     */
    private $customShippingRateFactory;

    /**
     * @var \Amasty\AdvancedMSI\Model\ResourceModel\SourceCustomShippingRate
     */
    private $sourceCustomShippingRateResource;

    /**
     * @var \Amasty\AdvancedMSI\Model\ResourceModel\SourceCustomShippingRate\CollectionFactory
     */
    private $collectionFactory;

    public function __construct(
        \Amasty\AdvancedMSI\Model\SourceCustomShippingRateFactory $customShippingRateFactory,
        \Amasty\AdvancedMSI\Model\ResourceModel\SourceCustomShippingRate $sourceCustomShippingRateResource,
        \Amasty\AdvancedMSI\Model\ResourceModel\SourceCustomShippingRate\CollectionFactory $collectionFactory
    ) {
        $this->customShippingRateFactory = $customShippingRateFactory;
        $this->sourceCustomShippingRateResource = $sourceCustomShippingRateResource;
        $this->collectionFactory = $collectionFactory;
    }

    /**
     * @inheritDoc
     */
    public function getRatesBySourceCode($sourceCode): array
    {
        $sourceCustomShippingRates = $this->collectionFactory->create()->addFieldToFilter(
            SourceCustomShippingRateInterface::SOURCE_CODE,
            $sourceCode
        )->getItems();

        $result = [];
        /** @var \Amasty\AdvancedMSI\Api\Data\SourceCustomShippingRateInterface $sourceCustomShippingRate */
        foreach ($sourceCustomShippingRates as $sourceCustomShippingRate) {
            $result[$sourceCustomShippingRate->getShippingRateCode()] = $sourceCustomShippingRate;
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function save(string $sourceCode, array $sourceCustomShippingRates): array
    {
        $this->sourceCustomShippingRateResource->deleteBySourceCode($sourceCode);
        foreach ($sourceCustomShippingRates as $sourceCustomShippingRate) {
            try {
                $this->sourceCustomShippingRateResource->save($sourceCustomShippingRate);
            } catch (\Exception $e) {
                throw new CouldNotSaveException(__($e->getMessage()), $e);
            }
        }

        return $sourceCustomShippingRates;
    }

    /**
     * @inheritDoc
     */
    public function getEmptySourceCustomShippingRateModel()
        : \Amasty\AdvancedMSI\Api\Data\SourceCustomShippingRateInterface
    {
        return $this->customShippingRateFactory->create();
    }
}
