<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\Repository;

use Amasty\AdvancedMSI\Api\Data\ReservedProductsInterface;
use Amasty\AdvancedMSI\Api\ReservedProductsRepositoryInterface;
use Amasty\AdvancedMSI\Model\ReservedProductsFactory;
use Amasty\AdvancedMSI\Model\ResourceModel\ReservedProducts as ReservedProductsResource;
use Amasty\AdvancedMSI\Model\ResourceModel\ReservedProducts\CollectionFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class ReservedProductsRepository implements ReservedProductsRepositoryInterface
{
    /**
     * @var ReservedProductsFactory
     */
    private $reservedProductsFactory;

    /**
     * @var ReservedProductsResource
     */
    private $reservedProductsResource;

    /**
     * @var CollectionFactory
     */
    private $reservedProductsCollectionFactory;

    /**
     * Model data storage
     *
     * @var array
     */
    private $_reservedProducts;

    public function __construct(
        ReservedProductsFactory $reservedProductsFactory,
        ReservedProductsResource $reservedProductsResource,
        CollectionFactory $reservedProductsCollectionFactory
    ) {
        $this->reservedProductsFactory = $reservedProductsFactory;
        $this->reservedProductsResource = $reservedProductsResource;
        $this->reservedProductsCollectionFactory = $reservedProductsCollectionFactory;
    }

    /**
     * @inheritdoc
     */
    public function save(ReservedProductsInterface $reservedProduct): ReservedProductsInterface
    {
        try {
            $this->reservedProductsResource->save($reservedProduct);
            unset($this->_reservedProducts[$reservedProduct->getItemId()]);
        } catch (\Exception $e) {
            if ($reservedProduct->getItemId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save reservedProduct with ID %1. Error: %2',
                        [$reservedProduct->getItemId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new reservedProduct. Error: %1', $e->getMessage()));
        }

        return $reservedProduct;
    }

    /**
     * @inheritdoc
     */
    public function getById($id): ReservedProductsInterface
    {
        if (!isset($this->_reservedProducts[$id])) {
            /** @var \Amasty\AdvancedMSI\Model\ReservedProducts $reservedProduct */
            $reservedProduct = $this->reservedProductsFactory->create();
            $this->reservedProductsResource->load($reservedProduct, $id);
            if (!$reservedProduct->getItemId()) {
                throw new NoSuchEntityException(__('ReservedProduct with specified ID "%1" not found.', $id));
            }
            $this->_reservedProducts[$id] = $reservedProduct;
        }

        return $this->_reservedProducts[$id];
    }

    /**
     * @inheritdoc
     */
    public function delete(ReservedProductsInterface $reservedProduct): bool
    {
        try {
            $this->reservedProductsResource->delete($reservedProduct);
            unset($this->_reservedProducts[$reservedProduct->getItemId()]);
        } catch (\Exception $e) {
            if ($reservedProduct->getItemId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove reservedProduct with ID %1. Error: %2',
                        [$reservedProduct->getItemId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove withConsent. Error: %1', $e->getMessage()));
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteById($id): bool
    {
        $reservedProductModel = $this->getById($id);
        $this->delete($reservedProductModel);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteByOrderId($orderId): bool
    {
        $reservedProducts = $this->getByOrderId($orderId);

        foreach ($reservedProducts as $product) {
            $this->delete($product);
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function getByOrderId($orderId): array
    {
        /** @var \Amasty\AdvancedMSI\Model\reservedProductsCollection $reservedProductsCollection */
        $reservedProductsCollection = $this->reservedProductsCollectionFactory->create();
        $reservedProducts = $reservedProductsCollection->addFieldToFilter('order_id', $orderId);

        if (!$reservedProducts) {
            throw new NoSuchEntityException(__('ReservedProducts with specified order ID "%1" not found.', $orderId));
        }

        return $reservedProducts->getItems();
    }
}
