<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model\Import\Source\Validation;

use Amasty\Base\Model\Import\AbstractImport;
use Amasty\Base\Model\Import\Validation\Validator;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingError;

class Basic extends Validator implements \Amasty\Base\Model\Import\Validation\ValidatorInterface
{
    public const ERROR_COL_SOURCE_CODE = 'errorSourceCodeEmpty';
    public const ERROR_COL_SOURCE_TITLE = 'errorSourceTitleEmpty';
    public const ERROR_COL_COUNTRY = 'errorCountryEmpty';
    public const ERROR_COL_POSTCODE = 'errorPostCodeEmpty';
    public const ERROR_SOURCE_CODE_ALREADY_EXIST = 'errorSourceCodeAlreadyExist';

    /**
     * @var \Magento\InventoryApi\Api\SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var \Magento\InventoryApi\Api\Data\SourceInterfaceFactory
     */
    private $sourceFactory;

    /**
     * @var \Magento\InventoryApi\Model\SourceValidatorInterface
     */
    private $sourceValidator;

    public function __construct(
        \Magento\Framework\DataObject $validationData,
        \Magento\InventoryApi\Api\SourceRepositoryInterface $sourceRepository,
        \Magento\InventoryApi\Api\Data\SourceInterfaceFactory $sourceFactory,
        \Magento\InventoryApi\Model\SourceValidatorInterface $sourceValidator
    ) {
        parent::__construct($validationData);
        $this->sourceRepository = $sourceRepository;
        $this->sourceFactory = $sourceFactory;
        $this->sourceValidator = $sourceValidator;
    }

    /**
     * @var array
     */
    protected $messageTemplates = [
        self::ERROR_COL_SOURCE_CODE => '<b>Error!</b> Source Code Field Is Empty',
        self::ERROR_COL_SOURCE_TITLE => '<b>Error!</b> Source Name Field Is Empty',
        self::ERROR_COL_COUNTRY => '<b>Error!</b> Country Field Is Empty',
        self::ERROR_COL_POSTCODE => '<b>Error!</b> Post Code Field Is Empty',
        self::ERROR_SOURCE_CODE_ALREADY_EXIST => '<b>Error!</b> Source Code Already Exist'
    ];

    /**
     * @inheritdoc
     */
    public function validateRow(array $rowData, $behavior)
    {
        $this->errors = [];

        if (empty($rowData['source_code'])) {
            $this->errors[self::ERROR_COL_SOURCE_CODE] = ProcessingError::ERROR_LEVEL_CRITICAL;
        } else {
            try {
                $this->sourceRepository->get($rowData['source_code']);
                $this->errors[self::ERROR_SOURCE_CODE_ALREADY_EXIST] = ProcessingError::ERROR_LEVEL_CRITICAL;
            } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                //everything is okay
                null;
            }
        }

        $validationResult = $this->sourceValidator->validate(
            $this->sourceFactory->create()->setData($rowData)
        );

        if ($validationResult->getErrors()) {
            foreach ($validationResult->getErrors() as $error) {
                $this->addRuntimeError($error, ProcessingError::ERROR_LEVEL_CRITICAL);
            }
        }

        return parent::validateResult();
    }
}
