<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Advanced MSI for Magento 2
 */

namespace Amasty\AdvancedMSI\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

/**
 * Scope config Provider model
 */
class ConfigProvider
{
    public const PATH_PREFIX = 'advanced_msi';
    public const XPATH_ENABLED = 'general/enabled';
    public const DEFAULT_ALGORITHM = 'delivery/default_algorithm';
    public const COMBINED_ALGORITHMS_VALUE = 'algorithms/combined_value_';
    public const ROUND_DISTANCE = 'delivery/round_distance';
    public const SHIPPING_COST_ON_SOURCE_ADDRESS = 'shipping/shipping_address_depend';
    public const EMAIL_NOTIFICATIONS_PATH = 'email_notifications';

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var array
     */
    private $combinedAlgorithms;

    public function __construct(
        ScopeConfigInterface $scopeConfig
    ) {
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * An alias for scope config with default scope type SCOPE_STORE
     */
    public function getValue(string $key, string $scopeType = ScopeInterface::SCOPE_STORE): ?string
    {
        return $this->scopeConfig->getValue(self::PATH_PREFIX . '/' . $key, $scopeType);
    }

    public function isEnabled(): bool
    {
        return (bool)$this->getValue(self::XPATH_ENABLED, ScopeConfigInterface::SCOPE_TYPE_DEFAULT);
    }

    public function getDefaultAlgorithm(): string
    {
        return (string)$this->getValue(self::DEFAULT_ALGORITHM, ScopeConfigInterface::SCOPE_TYPE_DEFAULT);
    }

    public function getRoundDistance(): int
    {
        return (int)$this->getValue(self::ROUND_DISTANCE, ScopeConfigInterface::SCOPE_TYPE_DEFAULT) * 1000;
    }

    public function getShippingCostOnSourceEnabled(): bool
    {
        return (bool)$this->getValue(self::SHIPPING_COST_ON_SOURCE_ADDRESS, ScopeConfigInterface::SCOPE_TYPE_DEFAULT);
    }

    public function getEmailTemplateId(string $code): string
    {
        return (string)$this->getValue(
            self::EMAIL_NOTIFICATIONS_PATH . '/' . $code,
            ScopeConfigInterface::SCOPE_TYPE_DEFAULT
        );
    }

    public function getCombinedAlgorithmValue(): array
    {
        if ($this->combinedAlgorithms == null) {
            $data = $this->getValue('delivery/combined_value');

            if (!empty($data)) {
                $data = json_decode($data, true);
                foreach ($data as $key => $value) {
                    if (preg_match('/combined_value_(.*)_(is_active|priority)/is', $key, $match)) {
                        $this->combinedAlgorithms[$match[1]][$match[2]] = $value['value'];
                    }
                }
            } else {
                $this->combinedAlgorithms = [];
            }
        }

        return $this->combinedAlgorithms;
    }
}
