<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Admin Actions Log Schedule Update
 */

namespace Amasty\AdminActionsLogScheduleUpdate\Model;

use Amasty\AdminActionsLog\Api\Logging\MetadataInterface;
use Amasty\AdminActionsLog\Api\Logging\MetadataInterfaceFactory;
use Amasty\AdminActionsLog\Logging\ActionFactory;
use Amasty\AdminActionsLogScheduleUpdate\Logging\Schedule\CollectorProvider;

class StagingLoggingProcessor
{
    public const UPDATED_BY_SCHEDULE = 'updated by schedule';

    /**
     * @var ActionFactory
     */
    private $actionFactory;

    /**
     * @var MetadataInterfaceFactory
     */
    private $metadataFactory;

    /**
     * @var CollectorProvider
     */
    private $collectorProvider;

    /**
     * @var StagingObjectFactory
     */
    private $objectFactory;

    /**
     * @var array
     */
    private $entitiesData = [];

    /**
     * @var string
     */
    private $storageEntityKey = '';

    public function __construct(
        ActionFactory $actionFactory,
        MetadataInterfaceFactory $metadataFactory,
        CollectorProvider $collectorProvider,
        StagingObjectFactory $objectFactory
    ) {
        $this->actionFactory = $actionFactory;
        $this->metadataFactory = $metadataFactory;
        $this->collectorProvider = $collectorProvider;
        $this->objectFactory = $objectFactory;
    }

    public function saveEntities(): void
    {
        if (!$this->objectFactory->isModuleEnabled()) {
            return;
        }

        $updateRepository = $this->objectFactory->getUpdateRepositoryObject();
        $getNotIndexedEntities = $this->objectFactory->getGetNotIndexedEntitiesObject();
        $stagingList = $this->objectFactory->getStagingListObject();
        $versionHistory = $this->objectFactory->getVersionHistoryObject();

        $currentVersionId = $updateRepository->getVersionMaxIdByTime(strtotime('now'));
        if (!empty($currentVersionId) && $currentVersionId !== (string)$versionHistory->getCurrentId()) {
            $oldVersionId = $versionHistory->getCurrentId();
            foreach ($stagingList->getEntityTypes() as $entityType) {
                $entityIds = $getNotIndexedEntities->execute(
                    $entityType,
                    $oldVersionId,
                    $currentVersionId
                );
                $collector = $this->collectorProvider->getCollectorByType((string)$entityType);
                if ($entityIds && $collector) {
                    $this->entitiesData[$entityType] = $collector->collect($entityIds);
                }
            }
        }
    }

    public function logData(): void
    {
        if (!$this->objectFactory->isModuleEnabled()) {
            return;
        }

        foreach ($this->entitiesData as $entityType => $entities) {
            if ($collector = $this->collectorProvider->getCollectorByType((string)$entityType)) {
                foreach ($entities as $entityId => $oldEntity) {
                    $this->storageEntityKey = spl_object_id($oldEntity) . '.before';
                    $this->executeLogging($oldEntity, MetadataInterface::EVENT_SAVE_BEFORE);
                    $newEntities = $collector->collect([$entityId]);
                    $this->executeLogging(array_shift($newEntities), MetadataInterface::EVENT_SAVE_AFTER);
                }
            }
        }
    }

    private function executeLogging($loggingObject, string $eventName): void
    {
        $metadata = $this->metadataFactory->create([
            'eventName' => $eventName,
            'loggingObject' => $loggingObject,
            'storageEntityKey' => $this->storageEntityKey
        ]);
        $actionHandler = $this->actionFactory->create($metadata);
        $actionHandler->execute();
    }
}
