<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Shipping & Payment By Customer Groups for Magento 2
 */
namespace Amasty\Methods\Block\Adminhtml\Payment\Edit\Tab;

use Amasty\Methods\Controller\Adminhtml\RegistryConstants;
use Amasty\Methods\Model\Structure;
use Magento\Backend\Block\Template\Context;
use Magento\Backend\Block\Widget\Form\Generic;
use Magento\Backend\Block\Widget\Tab\TabInterface;
use Magento\Customer\Api\GroupManagementInterface;
use Magento\Framework\App\Config\Initial;
use Magento\Framework\Convert\DataObject;
use Magento\Framework\Data\FormFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Phrase;
use Magento\Framework\Registry;
use Magento\Payment\Model\Config;
use Magento\Store\Model\System\Store;

class General extends Generic implements TabInterface
{

    /**
     * @var Store
     */
    protected $_store;

    /**
     * @var Config
     */
    protected $_paymentConfig;

    /**
     * @var GroupManagementInterface
     */
    protected $_groupManagement;

    /**
     * @var DataObject
     */
    protected $_converter;

    /**
     * @var array
     */
    protected $_customerGroupsArray;

    /**
     * @var Initial
     */
    protected $_initialConfig;

    public function __construct(
        Context $context,
        Registry $registry,
        FormFactory $formFactory,
        Store $store,
        Config $paymentConfig,
        GroupManagementInterface $groupManagement,
        DataObject $converter,
        Initial $initialConfig,
        array $data = []
    ) {
        parent::__construct($context, $registry, $formFactory, $data);
        $this->_store = $store;
        $this->_paymentConfig = $paymentConfig;
        $this->_groupManagement = $groupManagement;
        $this->_converter = $converter;
        $this->_initialConfig = $initialConfig;
    }

    /**
     * @return Phrase
     */
    public function getTabLabel()
    {
        return __('Payment Methods');
    }

    /**
     * Prepare title for tab
     *
     * @return Phrase
     * @codeCoverageIgnore
     */
    public function getTabTitle()
    {
        return __('Payment Methods');
    }

    /**
     * Returns status flag about this tab can be showed or not
     *
     * @return bool
     * @codeCoverageIgnore
     */
    public function canShowTab()
    {
        return true;
    }

    /**
     * Returns status flag about this tab hidden or not
     *
     * @return bool
     * @codeCoverageIgnore
     */
    public function isHidden()
    {
        return false;
    }

    /**
     * @return array
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function getCustomerGroupsArray()
    {
        if (!$this->_customerGroupsArray) {
            $groups = array_merge(
                [$this->_groupManagement->getNotLoggedInGroup()],
                $this->_groupManagement->getLoggedInGroups()
            );
            $this->_customerGroupsArray = $this->_converter->toOptionArray($groups, 'id', 'code');
        }
        return $this->_customerGroupsArray;
    }

    /**
     * @return General
     * @throws LocalizedException
     */
    protected function _prepareForm()
    {
        $model = $this->_coreRegistry->registry(RegistryConstants::CURRENT_AMASTY_METHODS_PAYMENT);

        /** @var \Magento\Framework\Data\Form $form */
        $form = $this->_formFactory->create();
        $form->setHtmlIdPrefix('amasty_methods_');

        $fieldset = $form->addFieldset('scope_fieldset', ['legend' => __('Scopes')]);

        $fieldset->addField(
            'website_id',
            'select',
            [
                'label' => __('Current Scope'),
                'title' => __('Current Scope'),
                'name' => 'website_id',
                'required' => true,
                'options' => $this->_store->getWebsiteOptionHash(true)
            ]
        );

        $fieldset = $form->addFieldset('methods_fieldset', ['legend' => __('Payment Methods')]);

        $fieldset->addField(
            Structure::VAR_RESTRICT_METHOD,
            'select',
            [
                'label' => __('Action for selected groups'),
                'title' => __('Action for selected groups'),
                'name' => Structure::VAR_RESTRICT_METHOD,
                'options' => [0 => __('Allow'), 1 => __('Deny')]
            ]
        );

        $methods = $this->_initialConfig->getData('default')['payment'];

        foreach ($methods as $methodCode => $method) {
            if (empty($method['title'])) {
                $method['title'] = $methodCode;
            }

            if (isset($method['title'])) {
                $fieldset->addField(
                    'payment_method_' . $methodCode,
                    'multiselect',
                    [
                        'label' => $method['title'],
                        'title' => $method['title'],
                        'name' => $model->getObjectCode() . '[' . $methodCode . ']',
                        'values' => $this->getCustomerGroupsArray()
                    ]
                );
            }
        }

        $form->setValues($model->getData());

        $this->setForm($form);

        return parent::_prepareForm();
    }

    /**
     * @param $html
     * @return string
     */
    protected function _afterToHtml($html)
    {
        $html = parent::_afterToHtml($html);
        $html .= "<script>
            require(['prototype'], function(){
                $('amasty_methods_website_id').on('change', function(){
                    jQuery('body').trigger('processStart');
                    document.location.href = '" . $this->getUrl('amasty_methods/payment/index', [
                'website_id' => '_website_id'
            ]) . "'.replace('_website_id', this.value);
                })
            })
        </script>";
        return $html;
    }
}
