<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Mega Menu GraphQl for Magento 2 (System)
 */

namespace Amasty\MegaMenuGraphQl\Test\GraphQl;

use Amasty\MegaMenu\Api\ItemRepositoryInterface;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Model\ResourceModel\Category\Collection as CategoryCollection;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GetAmMegaMenuAllTest extends GraphQlAbstract
{
    public const MAIN_RESPONSE_KEY = 'amMegaMenuAll';
    public const ITEMS_RESPONSE_KEY = 'items';

    /**
     * @var ItemRepositoryInterface
     */
    private $itemRepository;

    /**
     * @var CategoryCollection
     */
    private $categoryCollection;

    /**
     * @var CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var Uid
     */
    private $idEncoder;

    protected function setUp(): void
    {
        parent::setUp();

        $this->itemRepository = Bootstrap::getObjectManager()->get(ItemRepositoryInterface::class);
        $this->categoryCollection = Bootstrap::getObjectManager()->create(CategoryCollection::class);
        $this->categoryRepository = Bootstrap::getObjectManager()->create(CategoryRepositoryInterface::class);
        $this->searchCriteriaBuilder = Bootstrap::getObjectManager()->create(SearchCriteriaBuilder::class);
        $this->idEncoder = Bootstrap::getObjectManager()->create(Uid::class);
    }

    /**
     * @group amasty_megamenu
     *
     * @magentoApiDataFixture Amasty_MegaMenuGraphQl::Test/GraphQl/_files/create_categories.php
     * @magentoApiDataFixture Amasty_MegaMenuGraphQl::Test/GraphQl/_files/create_mega_menu_item.php
     */
    public function testAmMegaMenuAll()
    {
        $custMegaItemName = 'Test Name Mega Menu Custom Item';
        $custItemLink = 'amasty_link.html';
        $catName = 'Special Mega Menu Categ Parent';
        $catChildName = 'Special Mega Menu Categ Child';

        $categoryResult = $this->categoryCollection->addFieldToFilter('name', $catName)->getFirstItem();
        $category = $this->categoryRepository->get((int)$categoryResult->getId());
        $categoryValues = [
            'name' => $catName,
            'content' => '{{child_categories_content}}',
            'desktop_font' => '700',
            'has_active' => false,
            'icon' => '',
            'is_active' => false,
            'id' => "category-node-" . $category->getId(),
            'is_category' => true,
            'is_parent_active' => true,
            'level' => 0,
            'mobile_font' => '700',
            'parent_id' => (string)$category->getParentId(),
            'parent_uid' => $this->idEncoder->encode((string)$category->getParentId()),
            'path' => $category->getPath(),
            'subcategories_position' => 0,
            'submenu_type' => 0,
            'uid' => $this->idEncoder->encode((string)$category->getId()),
        ];

        $categoryChild = $this->categoryRepository->get((int)$category->getChildren());
        $categoryChildValues = [
            'name' => $catChildName,
            'content' => '',
            'desktop_font' => '700',
            'has_active' => false,
            'icon' => '',
            'is_active' => false,
            'id' => "category-node-" . $categoryChild->getId(),
            'is_category' => true,
            'is_parent_active' => true,
            'level' => 1,
            'mobile_font' => '700',
            'parent_id' => 'category-node-' . $categoryChild->getParentId(),
            'parent_uid' => $this->idEncoder->encode((string)$categoryChild->getParentId()),
            'path' => $categoryChild->getPath(),
            'subcategories_position' => 0,
            'submenu_type' => 0,
            'uid' => $this->idEncoder->encode((string)$categoryChild->getId())
        ];

        $searchCriteria = $this->searchCriteriaBuilder->addFilter('name', $custMegaItemName)->create();
        $custItem = $this->itemRepository->getList($searchCriteria)->getItems()[0];
        $custItemName = $custItem->getName();
        $custItemValues = [
            'name' => $custItemName,
            'content' => '',
            'desktop_font' => '12',
            'has_active' => false,
            'icon' => '',
            'is_active' => false,
            'id' => "custom-node-" . $custItem->getEntityId(),
            'is_category' => false,
            'is_parent_active' => true,
            'label' => $custItem->getLabel(),
            'label_background_color' => $custItem->getLabelBackgroundColor(),
            'label_text_color' => $custItem->getLabelTextColor(),
            'level' => 0,
            'mobile_font' => '10',
            'parent_id' => '2',
            'parent_uid' => $this->idEncoder->encode('2'),
            'status' => $custItem->getStatus(),
            'subcategories_position' => 0,
            'submenu_type' => 0,
            'uid' => $this->idEncoder->encode('custom-node-' . $custItem->getEntityId()),
            'url' => $custItemLink,
            'width' => 2
        ];

        $response = $this->graphQlQuery($this->getQuery());

        // assert main response key and fields values
        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertArrayHasKey(self::ITEMS_RESPONSE_KEY, $response[self::MAIN_RESPONSE_KEY]);

        $assertedCategory = false;
        $assertedCategoryChild = false;
        $assertedCustomItem = false;

        foreach ($response[self::MAIN_RESPONSE_KEY][self::ITEMS_RESPONSE_KEY] as $key => $item) {
            if ($item['name'] == $catName) {
                $this->assertResponseFields(
                    $response[self::MAIN_RESPONSE_KEY][self::ITEMS_RESPONSE_KEY][$key],
                    $categoryValues
                );
                $this->assertNull($item['children']);
                $this->assertNull($item['column_count']);
                $this->assertNull($item['label']);
                $this->assertNull($item['label_background_color']);
                $this->assertNull($item['label_text_color']);
                $this->assertNull($item['status']);
                $this->assertNull($item['width']);
                $this->assertStringContainsString($item['url'], $category->getUrl());
                $assertedCategory = true;
            }

            if ($item['name'] == $catChildName) {
                $this->assertResponseFields(
                    $response[self::MAIN_RESPONSE_KEY][self::ITEMS_RESPONSE_KEY][$key],
                    $categoryChildValues
                );
                $this->assertNull($item['children']);
                $this->assertNull($item['column_count']);
                $this->assertNull($item['label']);
                $this->assertNull($item['label_background_color']);
                $this->assertNull($item['label_text_color']);
                $this->assertNull($item['status']);
                $this->assertNull($item['width']);
                $this->assertStringContainsString($item['url'], $categoryChild->getUrl());
                $assertedCategoryChild = true;
            }

            if ($item['name'] == $custItemName) {
                $this->assertResponseFields(
                    $response[self::MAIN_RESPONSE_KEY][self::ITEMS_RESPONSE_KEY][$key],
                    $custItemValues
                );
                $this->assertNull($item['children']);
                $this->assertNull($item['column_count']);
                $this->assertNull($item['path']);
                $assertedCustomItem = true;
            }
        }

        $this->assertTrue(
            $assertedCategory,
            "Category \"$catName\" has not valid values or not found in response."
        );
        $this->assertTrue(
            $assertedCategoryChild,
            "Child category \"$catChildName\" has not valid values or not found in response."
        );
        $this->assertTrue(
            $assertedCustomItem,
            "Custom Item \"$custItemName\" has not valid values or not found in response."
        );
    }

    private function getQuery(): string
    {
        return <<<QUERY
query {
  amMegaMenuAll {
    items{
      name
      children{
        id
      }
      column_count
      content
      desktop_font
      has_active
      icon
      is_active
      id
      is_category
      is_parent_active
      label
      label_background_color
      label_text_color
      level
      mobile_font
      parent_id
      parent_uid
      path
      status
      subcategories_position
      submenu_type
      uid
      url
      width
    }
  }
}
QUERY;
    }
}
