<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Jet Theme Sample Data for Magento 2 (System)
 */

namespace Amasty\JetThemeSampleData\Setup\Fixtures;

use Magento\CatalogSampleData\Model\Attribute;
use Magento\CatalogSampleData\Model\Category;
use Magento\ConfigurableSampleData;
use Magento\ConfigurableSampleData\Setup\Installer;
use Magento\Eav\Model\Config;
use Magento\Framework\Component\ComponentRegistrar;
use Magento\Framework\Filesystem\Directory\ReadFactory;
use Magento\Framework\Setup\SampleData\InstallerInterface;
use Magento\ImportExport\Model\Import;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\ImportExport\Model\Import\Source\CsvFactory;
use Magento\Indexer\Model\Indexer\CollectionFactory;
use Magento\Indexer\Model\IndexerFactory;
use Magento\ProductLinksSampleData\Model\ProductLink;

class Configurable extends Installer implements InstallerInterface
{
    /**
     * @var Attribute
     */
    private $attribute;

    /**
     * @var Category
     */
    private $category;

    /**
     * @var ProductLink
     */
    protected $productLinkSetup;

    /**
     * @var Import
     */
    private $importModel;

    /**
     * @var CsvFactory
     */
    private $csvSourceFactory;

    /**
     * @var ReadFactory
     */
    private $readFactory;

    /**
     * @var CollectionFactory
     */
    private $indexerCollectionFactory;

    /**
     * @var ComponentRegistrar
     */
    private $componentRegistrar;

    /**
     * @var Config
     */
    private $eavConfig;

    /**
     * @var IndexerFactory
     */
    private $indexerFactory;

    public function __construct(
        Attribute $attribute,
        Category $category,
        ProductLink $productLinkSetup,
        Config $eavConfig,
        Import $importModel,
        CsvFactory $csvSourceFactory,
        CollectionFactory $indexerCollectionFactory,
        ReadFactory $readFactory,
        ComponentRegistrar $componentRegistrar,
        IndexerFactory $indexerFactory
    ) {
        $this->attribute = $attribute;
        $this->category = $category;
        $this->productLinkSetup = $productLinkSetup;
        $this->eavConfig = $eavConfig;
        $this->importModel = $importModel;
        $this->csvSourceFactory = $csvSourceFactory;
        $this->indexerCollectionFactory = $indexerCollectionFactory;
        $this->readFactory = $readFactory;
        $this->componentRegistrar = $componentRegistrar;
        $this->indexerFactory = $indexerFactory;
    }

    /**
     * @return void
     */
    public function install(): void
    {
        $this->eavConfig->clear();
        $importModel = $this->importModel;
        $importModel->setData(
            [
                'entity' => 'catalog_product',
                'behavior' => 'append',
                'import_images_file_dir' => 'pub/media/catalog/product',
                Import::FIELD_NAME_VALIDATION_STRATEGY =>
                    ProcessingErrorAggregatorInterface::VALIDATION_STRATEGY_SKIP_ERRORS
            ]
        );

        $source = $this->csvSourceFactory->create(
            [
                'file' => 'fixtures/Configurable/products.csv',
                'directory' => $this->readFactory->create(
                    $this->componentRegistrar->getPath(ComponentRegistrar::MODULE, 'Amasty_JetThemeSampleData')
                )
            ]
        );

        $currentPath = getcwd();
        chdir(BP);
        $importModel->validateSource($source);
        $importModel->importSource();

        chdir($currentPath);

        $this->eavConfig->clear();

        $this->runReindex();
    }

    /**
     * @return void
     */
    private function runReindex(): void
    {
        $indexerCollection = $this->indexerCollectionFactory->create();
        $ids = $indexerCollection->getAllIds();
        foreach ($ids as $id) {
            $idx = $this->indexerFactory->create()->load($id);
            $idx->reindexAll($id);
        }
    }
}
