<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Jet Theme Backend for Magento 2 (System)
 */

namespace Amasty\JetTheme\Plugin\Magento\Checkout\CustomerData\AbstractItem;

use Amasty\JetTheme\Model\ConfigProvider;
use Magento\Checkout\CustomerData\AbstractItem;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\CurrencyInterface;
use Magento\Quote\Model\Quote\Item;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class AddSpecialPrice
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var CurrencyInterface
     */
    private $currency;

    public function __construct(
        ConfigProvider $configProvider,
        StoreManagerInterface $storeManager,
        LoggerInterface $logger,
        CurrencyInterface $currency
    ) {
        $this->configProvider = $configProvider;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
        $this->currency = $currency;
    }

    /**
     * @param AbstractItem $subject
     * @param array $result
     * @param Item $item
     */
    public function afterGetItemData(AbstractItem $subject, array $result, Item $item)
    {
        if (!$this->configProvider->isShowSpecialPriceOnMinicart() || !$this->hasSpecialPrice($item)) {
            return $result;
        }

        try {
            $convertedPrice = $this->convertToCurrentCurrency((float)$item->getProduct()->getPrice());
            $result['product_price_regular'] = $convertedPrice;
        } catch (\Exception $e) {
            $this->logger->critical($e);
        }

        return $result;
    }

    /**
     * @param Item $item
     * @return bool
     */
    private function hasSpecialPrice(Item $item): bool
    {
        if ($item->getProduct()->getSpecialPrice()) {
            return true;
        }

        if ($item->getProductType() == 'configurable') {
            foreach ($item->getChildren() as $child) {
                if ($child->getProduct()->getSpecialPrice()) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param float $price
     * @return string
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    private function convertToCurrentCurrency(float $price): string
    {
        /** @var \Magento\Store\Model\Store $store */
        $store = $this->storeManager->getStore();
        $currencyCode = $store->getCurrentCurrency()->getCode();
        $currency = $this->currency->getCurrency($currencyCode);

        if ($store->getBaseCurrencyCode() != $currencyCode) {
            $price = $store->getBaseCurrency()->convert($price, $currencyCode);
        }

        return $currency->toCurrency($price);
    }
}
