<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Jet Theme Backend for Magento 2 (System)
 */

namespace Amasty\JetTheme\Model\FooterLink;

use Amasty\JetTheme\Api\Data\FooterLinkInterface;
use Amasty\JetTheme\Api\Data\FooterLinkInterfaceFactory;
use Amasty\JetTheme\Api\Data\FooterLinkSearchResultsInterfaceFactory;
use Amasty\JetTheme\Api\FooterLinkRepositoryInterface;
use Amasty\JetTheme\Model\FooterLink\ResourceModel\FooterLink as ResourceFooterLink;
use Amasty\JetTheme\Model\FooterLink\ResourceModel\FooterLink\CollectionFactory as FooterLinkCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResults;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class FooterLinkRepository implements FooterLinkRepositoryInterface
{
    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var FooterLinkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var FooterLinkInterfaceFactory
     */
    private $footerLinkFactory;

    /**
     * @var ResourceFooterLink
     */
    private $resource;

    /**
     * @var FooterLinkCollectionFactory
     */
    private $footerLinkCollectionFactory;

    /**
     * @var FooterLinkInterface[]
     */
    private $loadedData = [];

    public function __construct(
        ResourceFooterLink $resource,
        FooterLinkInterfaceFactory $footerLinkFactory,
        FooterLinkCollectionFactory $footerLinkCollectionFactory,
        FooterLinkSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->footerLinkFactory = $footerLinkFactory;
        $this->footerLinkCollectionFactory = $footerLinkCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheirtDoc
     */
    public function save(FooterLinkInterface $footerLink): FooterLinkInterface
    {
        try {
            $this->resource->save($footerLink);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the footerLink: %1',
                $exception->getMessage()
            ));
        }

        return $footerLink;
    }

    /**
     * @inheirtDoc
     */
    public function get(int $footerLinkId): FooterLinkInterface
    {
        if (!isset($this->loadedData[$footerLinkId])) {
            $footerLinkModel = $this->footerLinkFactory->create();
            $this->resource->load($footerLinkModel, $footerLinkId);
            $this->loadedData[$footerLinkId] = $footerLinkModel;
            if (!$footerLinkModel->getId()) {
                throw new NoSuchEntityException(__('FooterLink with id "%1" does not exist.', $footerLinkId));
            }
        }

        return $this->loadedData[$footerLinkId];
    }

    /**
     * @inheirtDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResults
    {
        $collection = $this->footerLinkCollectionFactory->create();
        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);

        $items = $collection->getItems();

        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheirtDoc
     */
    public function delete(FooterLinkInterface $footerLink): bool
    {
        try {
            $footerLinkId = $footerLink->getId();
            $this->resource->delete($footerLink);
            if (isset($this->loadedData[$footerLinkId])) {
                unset($this->loadedData[$footerLinkId]);
            }
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the FooterLink: %1',
                $exception->getMessage()
            ));
        }

        return true;
    }

    /**
     * @inheirtDoc
     */
    public function deleteById(int $footerLinkId): bool
    {
        return $this->delete($this->get($footerLinkId));
    }
}
