<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\ResourceModel;

use Amasty\InstagramFeed\Api\Data\HashtagInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;

class Hashtag extends AbstractDb
{
    protected function _construct()
    {
        $this->_init(HashtagInterface::MAIN_TABLE, HashtagInterface::ID);
    }

    /**
     * @throws \Exception
     */
    public function addNewData(array $newHashtags): void
    {
        $this->getConnection()->beginTransaction();
        try {
            $this->getConnection()->insertMultiple($this->getMainTable(), $newHashtags);
            $this->getConnection()->commit();
        } catch (\Exception $e) {
            $this->getConnection()->rollBack();
            throw $e;
        }
    }

    /**
     * @throws LocalizedException
     */
    public function getCountHashtagsPerWeekLimit(): int
    {
        $select = $this->getConnection()->select()
            ->from(['hashtag' => $this->getMainTable()])
            ->where('hashtag.timestamp > DATE_SUB(CURRENT_TIMESTAMP(), INTERVAL 7 DAY)')
            ->reset(Select::COLUMNS)
            ->columns('COUNT(*)');

        return (int)$this->getConnection()->fetchOne($select);
    }

    /**
     * @throws LocalizedException
     */
    public function getNextDateToGenerateHashtag(): string
    {
        $select = $this->getConnection()->select()
            ->from(['hashtag' => $this->getMainTable()])
            ->where('hashtag.timestamp > DATE_SUB(CURRENT_TIMESTAMP(), INTERVAL 7 DAY)')
            ->reset(Select::COLUMNS)
            ->columns('(hashtag.timestamp + INTERVAL 7 DAY)')
            ->order('hashtag.timestamp ASC')
            ->limit(1);

        return (string)$this->getConnection()->fetchOne($select);
    }
}
