<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Video\VideoService;

use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\HttpCurl;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\ServiceData\VideoServiceDataInterface;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\ServiceData\VideoServiceDataInterfaceFactory;
use Magento\Framework\Exception\LocalizedException;

class VideoServiceResolver implements VideoServiceResolverInterface
{
    public const INSTAGRAM_VIDEO_POSITION = 'video_versions';

    private const INSTAGRAM_API_VIDEO_DATA = 'https://instagram.com/p/';
    private const USER_AGENT =
        'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/104.0.0.0 Safari/537.36';

    /**
     * @var HttpCurl
     */
    private $httpCurl;

    /**
     * @var VideoServiceDataInterfaceFactory
     */
    private $videoServiceDataInterfaceFactory;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        HttpCurl $httpCurl,
        VideoServiceDataInterfaceFactory $videoServiceDataInterfaceFactory,
        ConfigProvider $configProvider
    ) {
        $this->httpCurl = $httpCurl;
        $this->videoServiceDataInterfaceFactory = $videoServiceDataInterfaceFactory;
        $this->configProvider = $configProvider;
    }

    /**
     * @throws LocalizedException
     */
    public function getInstagramServiceVideoUrl(string $postUrl): VideoServiceDataInterface
    {
        $array = explode('/', rtrim($postUrl, '/'));
        $shortcode = end($array);

        $url = self::INSTAGRAM_API_VIDEO_DATA . $shortcode . '?' . $this->httpCurl->getQueryString(
            [
                '__a' => '1',
                '__d' => 'dis'
            ]
        );

        $cookieValue = sprintf('ds_user_id=0;sessionid=%s;', $this->configProvider->getSessionId());
        $result = $this->httpCurl->setHeaders(
            [
                'Cookie' => $cookieValue,
                'User-Agent' => self::USER_AGENT
            ]
        )->loadByUrl($url);

        $videoService = $this->videoServiceDataInterfaceFactory->create();
        if ($result && isset($result['items'][0][self::INSTAGRAM_VIDEO_POSITION])) {
            foreach ($result['items'][0][self::INSTAGRAM_VIDEO_POSITION] as $content) {
                $url = $content['url'];
                $videoService->setVideoUrl($url);
                $videoService->setVideoExt($this->getVideoExtension($url));
                break;
            }
        }

        return $videoService;
    }

    private function getVideoExtension(string $url): string
    {
        // phpcs:disable Magento2.Functions.DiscouragedFunction.Discouraged
        $path = parse_url($url, PHP_URL_PATH);

        // phpcs:disable Magento2.Functions.DiscouragedFunction.DiscouragedWithAlternative
        return pathinfo($path, PATHINFO_EXTENSION);
    }
}
