<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Video\Manager;

use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\ServiceData\VideoServiceDataInterface;
use Magento\Framework\Exception\ConfigurationMismatchException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\SecurityViolationException;
use Magento\Framework\Filesystem\Driver\Https;

class VideoFileChecker
{
    private const CONTENT_TYPE = 'type';
    private const CONTENT_SIZE = 'size';

    /**
     * @var Https
     */
    private $driverHttps;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        Https $driverHttps,
        ConfigProvider $configProvider
    ) {
        $this->driverHttps = $driverHttps;
        $this->configProvider = $configProvider;
    }

    /**
     * @throws SecurityViolationException|ConfigurationMismatchException
     * @throws LocalizedException
     */
    public function validateFile(VideoServiceDataInterface $videoData, ?int $storeId): void
    {
        $fileMetaData = $this->driverHttps->stat(
            preg_replace('(^https?://)', '', $videoData->getVideoUrl())
        );

        if ($fileMetaData[self::CONTENT_TYPE] !== sprintf('video/%s', $videoData->getVideoExt())) {
            throw new SecurityViolationException(
                __('Undefined content type: %1', $fileMetaData[self::CONTENT_TYPE])
            );
        }

        if (!$fileMetaData[self::CONTENT_SIZE]) {
            throw new LocalizedException(
                __('Content length could not be determined for Video URL: %1', $videoData->getVideoUrl())
            );
        }

        if ($fileMetaData[self::CONTENT_SIZE] / 1000000 > $this->configProvider->getMaxDownloadVideoSize($storeId)) {
            throw new ConfigurationMismatchException(
                __(
                    'The file size is larger (%1 MB) than the set in the admin panel (%2 MB)',
                    $fileMetaData[self::CONTENT_SIZE] / 1000000,
                    $this->configProvider->getMaxDownloadVideoSize($storeId)
                )
            );
        }
    }
}
