<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Video\Manager;

use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\Video\GetMediaDirectory;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\ServiceData\VideoServiceDataInterface;
use Laminas\Stdlib\Glob;
use Magento\Framework\Exception\ConfigurationMismatchException;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\SecurityViolationException;
use Magento\Framework\Filesystem\DriverInterface;

// TODO Decompose this class
class Manager implements ManagerInterface
{
    public const AMASTY_INSTAGRAM_VIDEO_FOLDER = 'amasty/instagramfeed';

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var GetMediaDirectory
     */
    private $getMediaDirectory;

    /**
     * @var DriverInterface
     */
    private $driver;

    /**
     * @var VideoFileChecker
     */
    private $videoFileChecker;

    public function __construct(
        ConfigProvider $configProvider,
        GetMediaDirectory $getMediaDirectory,
        DriverInterface $driver,
        VideoFileChecker $videoFileChecker
    ) {
        $this->configProvider = $configProvider;
        $this->getMediaDirectory = $getMediaDirectory;
        $this->driver = $driver;
        $this->videoFileChecker = $videoFileChecker;
    }

    /**
     * @throws FileSystemException
     * @throws SecurityViolationException
     * @throws ConfigurationMismatchException
     * @throws LocalizedException
     */
    public function save(VideoServiceDataInterface $videoData, int $postId, ?int $storeId): void
    {
        $this->videoFileChecker->validateFile($videoData, $storeId);
        $videoContent = $this->driver->fileGetContents($videoData->getVideoUrl());

        $directoryName = $this->getAbsolutePathOfInstagramDirectory($storeId);
        $this->getMediaDirectory->execute()->create($directoryName);

        $fileName = sprintf('%d.%s', $postId, $videoData->getVideoExt());
        $file = $this->getMediaDirectory->execute()->openFile($directoryName . DIRECTORY_SEPARATOR . $fileName, 'w');

        try {
            $file->lock();
            $file->write($videoContent);
        } finally {
            $file->unlock();
            $file->close();
        }
    }

    /**
     * @throws FileSystemException
     */
    public function getVideoFileSystemPath(int $postId, ?int $storeId): ?string
    {
        $path = $this->getAbsolutePathOfInstagramDirectory($storeId) . DIRECTORY_SEPARATOR . $postId;
        $pathsWithExt = Glob::glob($path . '.*');

        $filePath = null;
        foreach ($pathsWithExt as $pathWithExt) {
            if ($this->getMediaDirectory->execute()->isExist($pathWithExt)) {
                $filePath = $this->getMediaDirectory->execute()->getRelativePath($pathWithExt);
                break;
            }
        }

        return $filePath;
    }

    private function getVideoFolderRelativePath(?int $storeId): string
    {
        return self::AMASTY_INSTAGRAM_VIDEO_FOLDER
            . DIRECTORY_SEPARATOR
            . $this->configProvider->getUserId($storeId);
    }

    /**
     * @throws FileSystemException
     */
    private function getAbsolutePathOfInstagramDirectory(?int $storeId): string
    {
        return $this->getMediaDirectory->execute()->getAbsolutePath(
            $this->getVideoFolderRelativePath($storeId)
        );
    }
}
