<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Video;

use Amasty\InstagramFeed\Model\Instagram\GarbageCollectorInterface;
use Amasty\InstagramFeed\Model\Instagram\Video\Manager\Manager;
use Laminas\Stdlib\Glob;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Filesystem\Driver\File as FileDriver;
use Magento\Framework\Filesystem\Io\File;
use Psr\Log\LoggerInterface;

class DeleteOldVideoContent
{
    /**
     * @var GetMediaDirectory
     */
    private $getMediaDirectory;

    /**
     * @var FileDriver
     */
    private $driverFile;

    /**
     * @var File
     */
    private $file;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var GarbageCollectorInterface
     */
    private $garbageCollectorInterface;

    public function __construct(
        GetMediaDirectory $getMediaDirectory,
        FileDriver $driverFile,
        File $file,
        LoggerInterface $logger,
        GarbageCollectorInterface $garbageCollectorInterface
    ) {
        $this->getMediaDirectory = $getMediaDirectory;
        $this->driverFile = $driverFile;
        $this->file = $file;
        $this->logger = $logger;
        $this->garbageCollectorInterface = $garbageCollectorInterface;
    }

    public function execute(): void
    {
        try {
            $this->deleteUnusedInstagramFolders();
            $this->deleteUnusedVideoContent();
        } catch (FileSystemException $exception) {
            $this->logger->error(
                __(
                    'An error occurred while running GarbageCollector: %1',
                    $exception->getMessage()
                )
            );
        }
    }

    /**
     * @throws FileSystemException
     */
    private function deleteUnusedInstagramFolders(): void
    {
        $instagramDirectoryPath = $this->getInstagramDirectoryPath();
        if (!$this->driverFile->isExists($instagramDirectoryPath)) {
            return;
        }

        $paths = $this->driverFile->readDirectory($instagramDirectoryPath);

        $instagramFoldersInFileSystem = [];
        $actualInstagramAccountFoldersNames = array_keys($this->garbageCollectorInterface->getActualPostsIds());
        foreach ($paths as $path) {
            $instagramFoldersInFileSystem[] = $this->driverFile->getRelativePath($instagramDirectoryPath, $path);
        }

        $diffInstagramIds = $this->getDiffOfTwoArrays(
            $actualInstagramAccountFoldersNames,
            $instagramFoldersInFileSystem
        );
        if (!empty($diffInstagramIds)) {
            foreach ($diffInstagramIds as $instagramFolderForDelete) {
                $this->driverFile->deleteDirectory($instagramDirectoryPath . $instagramFolderForDelete);
            }
        }
    }

    /**
     * @throws FileSystemException
     */
    private function deleteUnusedVideoContent(): void
    {
        $instagramDirectoryPath = $this->getInstagramDirectoryPath();

        foreach ($this->garbageCollectorInterface->getActualPostsIds() as $key => $instagramPostsIds) {
            $path = $instagramDirectoryPath . $key . DIRECTORY_SEPARATOR;
            $videoAbsolutePaths = $this->driverFile->readDirectory($path);

            $videoInFileSystem = [];
            foreach ($videoAbsolutePaths as $videoAbsolutePath) {
                $pathInfo = $this->file->getPathInfo($videoAbsolutePath);
                $videoInFileSystem[] = $this->driverFile->getRelativePath($path, $pathInfo['filename']);
            }

            $diffVideoIds = $this->getDiffOfTwoArrays($videoInFileSystem, $instagramPostsIds);
            if (!empty($diffVideoIds)) {
                foreach ($diffVideoIds as $videoIdForDelete) {
                    $filesWithoutExt = Glob::glob($path . $videoIdForDelete . '.*');
                    if (!empty($filesWithoutExt)) {
                        foreach ($filesWithoutExt as $file) {
                            $this->driverFile->deleteFile($file);
                        }
                    }
                }
            }
        }
    }

    /**
     * @throws FileSystemException
     */
    private function getInstagramDirectoryPath(): string
    {
        return $this->getMediaDirectory->execute()->getAbsolutePath(
            Manager::AMASTY_INSTAGRAM_VIDEO_FOLDER
        ) . DIRECTORY_SEPARATOR;
    }

    private function getDiffOfTwoArrays(array $first, array $second): array
    {
        return array_merge(array_diff($first, $second), array_diff($second, $first));
    }
}
