<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram;

use Amasty\InstagramFeed\Api\Data\PostInterface;
use Amasty\InstagramFeed\Api\PostRepositoryInterface;
use Amasty\InstagramFeed\Model\Config\Source\Sorting;
use Amasty\InstagramFeed\Model\Sorting\GetSortingById;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

class Provider
{
    public const POST_COUNT = 20;

    /**
     * @var PostRepositoryInterface
     */
    private $postRepository;
    /**
     * @var Management
     */
    private $management;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var IsMainInstagramSettingsDifferent
     */
    private $isMainInstagramSettingsDifferent;

    /**
     * @var GetSortingById
     */
    private $getSortingById;

    public function __construct(
        PostRepositoryInterface $postRepository,
        Management $management,
        StoreManagerInterface $storeManager,
        IsMainInstagramSettingsDifferent $isMainInstagramSettingsDifferent,
        GetSortingById $getSortingById
    ) {
        $this->postRepository = $postRepository;
        $this->management = $management;
        $this->storeManager = $storeManager;
        $this->isMainInstagramSettingsDifferent = $isMainInstagramSettingsDifferent;
        $this->getSortingById = $getSortingById;
    }

    /**
     * @param $params
     * @return array|PostInterface[]
     */
    public function getPosts($params, ?int $storeId = null)
    {
        $sort = $params['sort'] ?? Sorting::NEWEST;
        $count = $params['count'] ?? self::POST_COUNT;

        if ($storeId === null) {
            try {
                $storeId = $this->getStoreId();
            } catch (NoSuchEntityException $e) {
                $storeId = Store::DEFAULT_STORE_ID;
            }
        }

        if (!$this->postRepository->isPostsExist($storeId)) {
            $this->management->update($storeId, true);
        }

        return $this->getFromRepository($storeId, (int)$sort, (int)$count);
    }

    /**
     * @param int $storeId
     * @param int $sort
     * @param int $count
     * @return array|PostInterface[]
     */
    private function getFromRepository(int $storeId, int $sort, int $count)
    {
        $sortField = $this->getSortingById->execute($sort);

        return $this->postRepository->getPosts($storeId, $sortField, $count);
    }

    /**
     * @throws NoSuchEntityException
     */
    private function getStoreId(): int
    {
        return $this->isMainInstagramSettingsDifferent->execute()
            ? (int)$this->storeManager->getStore()->getId()
            : Store::DEFAULT_STORE_ID;
    }
}
