<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Operation;

use Amasty\InstagramFeed\Api\Data\PostInterface;
use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\GarbageCollectorInterface;
use Amasty\InstagramFeed\Model\Instagram\Video\Manager\ManagerInterface;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\ServiceData\VideoServiceDataInterface;
use Amasty\InstagramFeed\Model\Instagram\Video\VideoService\VideoServiceResolverInterface;
use Amasty\InstagramFeed\Model\Source\MediaType;
use Magento\Framework\Exception\ConfigurationMismatchException;
use Magento\Framework\Exception\RemoteServiceUnavailableException;

class MediaPathResolver
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ManagerInterface
     */
    private $videoManagerInterface;

    /**
     * @var GarbageCollectorInterface
     */
    private $garbageCollectorInterface;

    /**
     * @var VideoServiceResolverInterface
     */
    private $videoServiceResolverInterface;

    public function __construct(
        ConfigProvider $configProvider,
        ManagerInterface $videoManagerInterface,
        GarbageCollectorInterface $garbageCollectorInterface,
        VideoServiceResolverInterface $videoServiceResolverInterface
    ) {
        $this->configProvider = $configProvider;
        $this->videoManagerInterface = $videoManagerInterface;
        $this->garbageCollectorInterface = $garbageCollectorInterface;
        $this->videoServiceResolverInterface = $videoServiceResolverInterface;
    }

    /**
     * @throws RemoteServiceUnavailableException|ConfigurationMismatchException
     */
    public function getMediaUrl(array $mediaData, ?int $storeId): ?string
    {
        if ($mediaData[PostInterface::MEDIA_TYPE] === MediaType::VIDEO) {
            $mediaData[PostInterface::MEDIA_URL] = null;
        }

        if ($this->configProvider->isVideoContentEnabled($storeId)
            && $mediaData[PostInterface::MEDIA_TYPE] === MediaType::VIDEO
            && !$this->videoManagerInterface->getVideoFileSystemPath((int)$mediaData[PostInterface::IG_ID], $storeId)
        ) {
            /** @var VideoServiceDataInterface $videoData */
            $videoData = $this->videoServiceResolverInterface->getInstagramServiceVideoUrl(
                $mediaData[PostInterface::PERMALINK]
            );
            if (!$videoData->getVideoUrl() || !$videoData->getVideoExt()) {
                throw new RemoteServiceUnavailableException(
                    __('The video file retrieval service returned an invalid response')
                );
            }
            $this->videoManagerInterface->save($videoData, (int)$mediaData[PostInterface::IG_ID], $storeId);
        }

        if ($this->configProvider->isVideoContentEnabled($storeId)
            && $mediaData[PostInterface::MEDIA_TYPE] === MediaType::VIDEO
        ) {
            $mediaData[PostInterface::MEDIA_URL]
                = $this->videoManagerInterface->getVideoFileSystemPath((int)$mediaData[PostInterface::IG_ID], $storeId);

            $this->garbageCollectorInterface->collect(
                (int)$mediaData[PostInterface::IG_ID],
                (int)$this->configProvider->getUserId($storeId)
            );
        }

        return $mediaData[PostInterface::MEDIA_URL]
            ?? $mediaData[PostInterface::THUMBNAIL_URL]
            ?? null;
    }
}
